# Copyright (C) 2002  Internet Software Consortium.
#
# Permission to use, copy, modify, and distribute this software for any
# purpose with or without fee is hereby granted, provided that the above
# copyright notice and this permission notice appear in all copies.
#
# THE SOFTWARE IS PROVIDED "AS IS" AND INTERNET SOFTWARE CONSORTIUM
# DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE INCLUDING ALL
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL
# INTERNET SOFTWARE CONSORTIUM BE LIABLE FOR ANY SPECIAL, DIRECT,
# INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING
# FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
# NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION
# WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.

# $Id: Connect.pm,v 1.6 2002/12/06 02:21:11 lidl Exp $

package ISC::CC::Group::Connect;

use strict;
use warnings;

use Carp;

use ISC::CC::Message;
use ISC::CC::Direct;
use ISC::CC::Group;
use ISC::Net::Connect;
use ISC::Net::LengthPrefix;

BEGIN {
    use Exporter ();
    our ($VERSION, @ISA, @EXPORT, @EXPORT_OK, %EXPORT_TAGS);

    $VERSION = do { my @r = (q$Revision: 1.6 $ =~ /\d+/g); sprintf "%d."."%02d" x $#r, @r};
    @ISA = qw(Exporter ISC::CC::Group);
    @EXPORT = qw();
    @EXPORT_OK = qw();
    %EXPORT_TAGS = ();
}

our @EXPORT_OK;

my $sock_count = 0;

sub new {
    my ($class, %args) = @_;

    $class = ref($class) || $class;

    my $self = bless({}, $class);

    my $name;
    if ($args{name}) {
	$name = $args{name};
    } else {
	$name = "CC::Group::Connect" . $sock_count++;
    }
    $self->{name} = $name;

    $self->SUPER::_init(%args);

    $self->{state} = $self->SUPER::GROUP_STATE_CONNECTING;

    $self->{cb} = $args{cb};
    $self->{error_cb} = $args{error_cb} if ($args{error_cb});
    $self->{connect_cb} = $args{connect_cb} if ($args{connect_cb});
    $self->{connect_error_cb} = $args{connect_error_cb} if ($args{connect_error_cb});

    my $proto = $args{Proto} || $ENV{ISC_MSGQ_PROTO} || "tcp";

    my %largs;
    if ($proto eq "tcp") {
	$largs{LocalPort} = $args{LocalPort} if ($args{LocalPort});
	$largs{LocalAddr} = $args{LocalAddr} if ($args{LocalAddr});
	$largs{PeerPort} = $args{PeerPort} || $ENV{ISC_MSGQ_PORT} || 9911;
	$largs{PeerAddr} = $args{PeerAddr} || $ENV{ISC_MSGQ_ADDR} || "127.0.0.1";
	$largs{ReuseAddr} = $args{ReuseAddr} if ($args{ReuseAddr});
	$largs{Proto} = "tcp";
    } elsif ($proto eq "local" || $proto eq "unix") {
	$largs{Peer} = $args{Peer} || $ENV{ISC_MSGQ_LOCAL} || "/tmp/msgq.sock";
	$largs{Proto} = "local";
    } else {
	croak "Protocol must be 'tcp', 'local', or 'unix' (default: tcp)";
    }

    $largs{cb} = \&_connect_cb;
    $largs{error_cb} = \&_connect_err;
    $largs{timeout} = $args{timeout} if ($args{timeout});
    $largs{name} = $self->{name};
    $largs{data} = $self;
    $largs{name} = "$name connect";
    
    $self->{sock} = new ISC::Net::Connect(%largs);

    return $self;
}

sub DESTROY {
    my ($self) = @_;

    if ($self->{sock}) {
	$self->{sock}->cancel;
    }

    $self->SUPER::DESTROY;
}

sub _connect_cb {
    my ($con) = @_;

    my $self = $con->data;

    $self->{lp} = new ISC::Net::LengthPrefix(cb => \&_msg_cb,
					     name => $self->name . " lp",
					     error_cb => \&_msg_err,
					     socket => $con->socket,
					     data => $self);

    $self->{state} = $self->SUPER::GROUP_STATE_LNAME;
    $self->{lp}->send(ISC::CC::Message::towire({ type => "getlname" }));
}

sub _connect_err {
    my ($con, $msg) = @_;

    my $self = $con->{data};

    $self->{connect_error_cb}->($self, $msg);
#   $con->socket->close;
    $con->cancel;
}

sub _msg_cb {
    my ($lp, $buf) = @_;

    my $self = $lp->data;

    $self->SUPER::_msg_cb($buf);
}

sub data {
    my ($self, $val) = @_;

    if (defined($val)) {
	$self->{data} = $val;
    }
    return $self->{data};
}

sub _msg_err {
    my ($lp, $msg) = @_;

    my $self = $lp->data;

    if ($self->{error_cb}) {
	$self->{error_cb}->($lp, $msg);
    }

    $self->{lp}->cancel;
    $self->{lp} = undef;
}

sub close {
    my ($self) = @_;

    $self->SUPER::_close;

    $self->{lp}->close;
    $self->{lp} = undef;

    $self->{sock}->cancel;
    $self->{sock} = undef;
}

1;
