# Copyright (C) 2002  Internet Software Consortium.
#
# Permission to use, copy, modify, and distribute this software for any
# purpose with or without fee is hereby granted, provided that the above
# copyright notice and this permission notice appear in all copies.
#
# THE SOFTWARE IS PROVIDED "AS IS" AND INTERNET SOFTWARE CONSORTIUM
# DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE INCLUDING ALL
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL
# INTERNET SOFTWARE CONSORTIUM BE LIABLE FOR ANY SPECIAL, DIRECT,
# INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING
# FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
# NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION
# WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.

# $Id: ReplicationSlave.pm,v 1.2 2002/12/06 02:21:12 lidl Exp $

package ISC::DB::ReplicationSlave;

use strict;
use warnings;

use Carp;

use Data::Dumper;
use DBI;

BEGIN {
    use Exporter ();
    our ($VERSION, @ISA, @EXPORT, @EXPORT_OK, %EXPORT_TAGS);

    $VERSION = do { my @r = (q$Revision: 1.2 $ =~ /\d+/g); sprintf "%d."."%02d" x $#r, @r};
    @ISA = qw(Exporter);
    @EXPORT = qw();
    @EXPORT_OK = qw();
    %EXPORT_TAGS = ();
}

our @EXPORT_OK;

sub new {
    my ($class, %args) = @_;

    $class = ref($class) || $class;
    my $self = bless({}, $class);

    my $name = $args{name};
    my $db = $args{db};

    $self->{db} = $db;
    $self->_init_sql;

    $self->{name} = $name;

    $self->_get_rep_info();

    $self->{curst} = 0;
    $self->{seq} = 0;
    $self->{log} = [];

    my $gc = $args{group_handle};
    if ($gc) {
	$self->{group} = $gc->join(group => $name . "_rep",
				   instance => "rep_seq");
    }

#    print Data::Dumper->Dump([$self], ["self"]);

    return $self;
}

sub _init_sql {
    my ($self) = @_;

    my $db = $self->db->db;

    $self->{sql_newst} = $db->prepare("SELECT nextval('rep_seq') AS version");
    $self->{sql_addlog} = $db->prepare("INSERT INTO rep_delta (repid, generation, to_vers, seq, delta, ts) VALUES (?, ?, ?, ?, ?, ?)");
    $self->{sql_rep_info} = $db->prepare("SELECT * FROM rep WHERE name=?");
}

sub name {
    my ($self) = @_;

    return $self->{name};
}

sub db {
    my ($self) = @_;

    return $self->{db};
}

sub type {
    my ($self) = @_;

    return $self->{type};
}

sub _get_rep_info {
    my ($self) = @_;

    my $db = $self->db->begin;

    my $sth = $self->{sql_rep_info};
    $sth->execute($self->{name});

    my $arr = $sth->fetchrow_hashref;
    if (!$arr) {
	die "No such replication name " . $self->{name};
    }

    if ($arr->{type} ne "S") {
	die "Replication set is not slave";
    }

    my %row = %{ $arr };

    $self->{repid} = $row{repid};
    $self->{generation} = $row{generation};
    $self->{type} = $row{type};
    $sth->finish;

    $self->db->commit;
}

sub newst {
    my ($self) = @_;

    my $db = $self->db->begin;

    my $sth = $self->{sql_newst};
    $sth->execute();
    my %arr = %{ $sth->fetchrow_hashref };
    $self->{curst} = $arr{version};
    $sth->finish;

    return $self->{curst};
}

sub curst {
    my ($self) = @_;

    return $self->{curst};
}

sub abort {
    my ($self) = @_;

    die "finish called before newst" if (!$self->{curst});

    $self->{curst} = 0;

    $self->db->abort;
}

sub addlog {
}

sub finish {
    my ($self) = @_;

    die "finish called before newst" if (!$self->{curst});

    my $idb = $self->db;

    $self->db->commit;

    $self->{curst} = 0;
}

1;
