#!/usr/bin/env perl
#
# Copyright (C) 2002, 2003  Internet Software Consortium.
#
# Permission to use, copy, modify, and distribute this software for any
# purpose with or without fee is hereby granted, provided that the above
# copyright notice and this permission notice appear in all copies.
#
# THE SOFTWARE IS PROVIDED "AS IS" AND INTERNET SOFTWARE CONSORTIUM
# DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE INCLUDING ALL
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL
# INTERNET SOFTWARE CONSORTIUM BE LIABLE FOR ANY SPECIAL, DIRECT,
# INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING
# FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
# NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION
# WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.

# $Id: write-zone.pl,v 1.20 2003/02/13 17:43:39 explorer Exp $

use lib '@prefix@';

use warnings;
use strict;

use Getopt::Long;
use Time::HiRes qw(gettimeofday);

use ISC::SRS::DB;

my $dbname;
my $dbuser;
my $dbpass;
my $outfile;
my $noapex;
my $cmd;
my $sleeptime;
my $ret = GetOptions("dbname=s" => \$dbname,
		     "dbuser=s" => \$dbuser,
		     "dbpass=s" => \$dbpass,
		     "out=s" => \$outfile,
		     "noapex" => \$noapex,
		     "cmd=s" => \$cmd,
		     "sleep=i" => \$sleeptime);

my %largs;
$largs{dbname} = $dbname || $ENV{ISC_SRS_DBNAME} || "srs";
$largs{dbuser} = $dbuser || $ENV{ISC_SRS_DBUSER} || "srs";
$largs{dbpass} = $dbpass || $ENV{ISC_SRS_DBPASS} || undef;

my $srs = new ISC::SRS::DB(replication => "master",
			   replication_name => "srs",
			   serializable => 1);

my $apex;
if ($noapex) {
    $apex = 'ASDASD/ASDASD';
} else {
    $apex = $srs->apex;
}

use constant INFINITY => "9223372036854775807";  # 2^63 - 1

my $last_st = -1;

sub check_for_change {
    return 1 if ($last_st == -1);

    my $dbh = $srs->begin;

    my $sth = $dbh->prepare("SELECT last_value FROM rep_seq");
    $sth->execute;
    my @ret = $sth->fetchrow_array;

    $sth->finish;
    $srs->commit;

    if ($last_st ne $ret[0]) {
	print STDERR "last_st = $last_st, new st = $ret[0]\n";
	return 1;
    } else {
	return 0;
    }
}

sub dumpzone {
    my $domains = {};
    my $hosts = {};
    my $host_used = {};

    my $start = gettimeofday;

    my $dbh = $srs->begin;

    if ($outfile) {
	open(OUT, ">$outfile");
    } else {
	*OUT = *STDOUT;
    }

    my ($oldsoa, $soa) = $srs->bump_soa_serial(apex => $apex);
    print OUT ";; apex = $apex\n";

    print STDERR "OLD SOA at st = $oldsoa->{st}\n";
    print STDERR "NEW SOA at st = $soa->{st}\n";
    print OUT "\@ SOA $soa->{origin} $soa->{contact} $soa->{serial} $soa->{refresh} $soa->{retry} $soa->{expire} $soa->{minimum}\n";

    print OUT "\$TTL $soa->{ttl}\n";

    $last_st = $soa->{st};

    #
    # Select all the hosts
    #
    print STDERR "Fetching NS records...\n";
    my $sth = $dbh->prepare("SELECT d.fqdn AS dfqdn, h.fqdn AS hfqdn FROM domain d, domain_host_map dhm, host h WHERE d.domain_id=dhm.domain_id AND dhm.host_id=h.host_id ORDER BY d.fqdn");
    $sth->execute();
    while (my @arr = $sth->fetchrow_array) {
	my $dn = $arr[0] . ".";
	$dn =~ s/\.$apex\.$//i;
	$dn = '@' if ($dn eq "$apex.");

	my $hn = $arr[1] . ".";
	$hn =~ s/\.$apex\.$//i;
	print "$dn NS $hn\n";
    }
    $sth->finish;

    #
    # Select the A and AAAA records
    #
    print STDERR "Generating A and AAA records...\n";
    $sth = $dbh->prepare("SELECT h.fqdn, hip.type, hip.addr FROM host h, host_ip_map hip WHERE h.host_id=hip.host_id ORDER BY h.fqdn");
    $sth->execute();
    while (my @arr = $sth->fetchrow_array) {
	my $hn = $arr[0] . ".";
	$hn =~ s/\.$apex\.$//i;
	$hn = '@' if ($hn eq "$apex.");

	if ($arr[1] eq "4") {
	    print OUT "$hn A $arr[2]\n";
	} else {
	    print OUT "$hn AAAA $arr[2]\n";
	}
    }
    $sth->finish;

    $srs->commit;

    print STDERR "Done.\n";
    print OUT "; EOF\n";
    if ($outfile) {
	close(OUT);
    }

    printf STDERR ("Total time to extract and write: %9.3f\n",
		   gettimeofday - $start);
}

while (1) {
    my $need_dump = check_for_change;
    if ($need_dump) {
	dumpzone;
	system($cmd) if ($cmd);
    } else {
	print STDERR "ST has not changed (still $last_st)\n";
    }

    last unless ($sleeptime);

    sleep($sleeptime);
}
