# Copyright (C) 2002  Internet Software Consortium.
#
# Permission to use, copy, modify, and distribute this software for any
# purpose with or without fee is hereby granted, provided that the above
# copyright notice and this permission notice appear in all copies.
#
# THE SOFTWARE IS PROVIDED "AS IS" AND INTERNET SOFTWARE CONSORTIUM
# DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE INCLUDING ALL
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL
# INTERNET SOFTWARE CONSORTIUM BE LIABLE FOR ANY SPECIAL, DIRECT,
# INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING
# FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
# NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION
# WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.

# $Id: DBI.pm,v 1.9 2002/12/06 02:21:09 lidl Exp $

package ISC::DBI;

use strict;
use warnings;

use Carp;
use DBI;

BEGIN {
    use Exporter ();
    our ($VERSION, @ISA, @EXPORT, @EXPORT_OK, %EXPORT_TAGS);

    $VERSION = do { my @r = (q$Revision: 1.9 $ =~ /\d+/g); sprintf "%d."."%02d" x $#r, @r};
    @ISA = qw(Exporter);
    @EXPORT = qw();
    @EXPORT_OK = qw();
    %EXPORT_TAGS = ();
}

our @EXPORT_OK;

sub new {
    my ($class, %args) = @_;

    $class = ref($class) || $class;
    my $self = bless({}, $class);

    my $dbname = $args{dbname};
    my $dbuser = $args{dbuser};
    my $dbpass = $args{dbpass};

    $self->{db} = DBI->connect("dbi:Pg:dbname=$dbname", $dbuser, $dbpass,
			       { AutoCommit => 0,
				 RaiseError => 1,
			         PrintError => 0 });
    croak "cannot connect to database" if (!$self->{db});

    $self->{dbname} = $dbname;
    $self->{dbuser} = $dbuser;

    $self->{lockiter} = 0;
    $self->{aborting} = 0;

    $self->_serializable if ($args{serializable});

    return $self;
}

sub dbname {
    my ($self) = @_;

    return $self->{dbname};
}

sub dbuser {
    my ($self) = @_;

    return $self->{dbuser};
}

sub db {
    my ($self) = @_;

    return $self->{db};
}

sub begin {
    my ($self) = @_;

    $self->{lockiter}++;

    return $self->{db};
}

sub commit {
    my ($self) = @_;

    if ($self->{lockiter} == 0) {
	croak "Commit without begin";
    }

    $self->{lockiter}--;

    if ($self->{lockiter} == 0) {
	if ($self->{aborting}) {
	    $self->{db}->rollback;
	    $self->{aborting} = 0;
	} else {
	    $self->{db}->commit;
	}
    }
}

sub rollback {
    my ($self) = @_;

    if ($self->{lockiter} == 0) {
	croak "abort without begin";
    }

    $self->{lockiter}--;

    if ($self->{lockiter} == 0) {
	$self->{db}->rollback;
	$self->{aborting} = 0;
    } else {
	$self->{aborting} = 1;
    }
}

sub in_transaction {
    my ($self) = @_;

    return $self->{lockiter};
}

sub _serializable {
    my ($self) = @_;

    my $sth = $self->db->prepare("SET SESSION CHARACTERISTICS AS TRANSACTION ISOLATION LEVEL SERIALIZABLE");
    $sth->execute();
    if ($sth->err) {
	croak "serializable: " . $sth->errstr;
    }
}

sub is_aborting {
    my ($self) = @_;

    return $self->{aborting};
}

sub disconnect {
    my ($self) = @_;

    $self->{db}->disconnect;
    $self->{db} = undef;
}

1;

__END__

=head1 NAME

ISC::DBI - open a postgresql database connection and provide begin/commit/abort

=head1 SYNOPSIS

 use ISC::DBI;

 my $db = new ISC::DBI(
    dbname => "somedb",
    dbuser => "someuser",
    dbpass => "somepass");

 my $dbname = $db->dbname;
 my $dbuser = $db->dbuser;
 my $dbhandle = $db->db;

 my $child = $db->begin;
 $db->commit;
 $db->rollback;

 my $is_in_transaction = $db->in_transaction;
 my $is_aborting = $db->is_aborting;

 $db->disconnect;

=head1 DESCRIPTION

This module is a wrapper around the standard C<DBI> package.  It also
provides C<begin>, C<commit>, and C<rollback> functions which can be
nested.

Each function called can issue a C<begin> and C<commit> call, and only the
final C<commit> (or C<abort>) will occur.  This lets the application provide
functions which can protect themselves by starting a transaction if needed,
and they equate to a counting mutex if a transaction is already pending.

If any call to C<rollback> is made, the whole transaction is marked as
aborting, and the final C<commit> or C<rollback> will actually be a
C<DBI::rollback> call.

=over

=item new ARGS

Create a new database handle.  C<dbname> is required.  The created handle
will be a PostgreSQL database handle, with AutoCommit off, RaiseError on,
and PrintError off.

C<dbname> specifies the database name to use when connecting.

C<dbuser> specifies the database user.

C<dbpass> specifies the database user's password.

=item dbname

Return the database name.

=item dbuser

Return the database user.

=item db

Return the DBI handle itself.

=item my $child = $db->begin

Returns a DBI handle which statements may be executed on.  When this
handle is no longer needed, C<$db->commit> or C<$db->rollback> should
be called, and $child no longer used.

=item $db->commit

Commit the trancation.  If this is the outermost C<commit> call,
C<DBI::commit> will be called (unless C<rollback> was called.  See below).

=item $db->rollback

Mark the transaction as needing rollback.  If this is the outermost block,
it will occur immediately, otherwise it will occur after the final C<commit>
or C<rollback> function is called.

=item $db->in_transcation

Returns true if a transaction is in progress.

=item $db->is_aborting

Returns true if a rollback is pending.

=item $db->disconnect

Disconnects from the database.

=back

=head1 AUTHOR

Written by Michael Graff for the Internet Software Consortium.

=head1 COPYRIGHT

Copyright (C) 2002 Internet Software Consortium.
