#!/usr/bin/env perl
#
# Copyright (C) 2002  Internet Software Consortium.
#
# Permission to use, copy, modify, and distribute this software for any
# purpose with or without fee is hereby granted, provided that the above
# copyright notice and this permission notice appear in all copies.
#
# THE SOFTWARE IS PROVIDED "AS IS" AND INTERNET SOFTWARE CONSORTIUM
# DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE INCLUDING ALL
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL
# INTERNET SOFTWARE CONSORTIUM BE LIABLE FOR ANY SPECIAL, DIRECT,
# INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING
# FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
# NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION
# WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.

# $Id: test-xaction.pl,v 1.34 2002/12/24 22:07:59 lidl Exp $

use lib '@prefix@';

use strict;
use warnings;

no strict 'refs';

use Data::Dumper;
$Data::Dumper::Indent = 1;
use Event qw(loop unloop);
use Time::HiRes qw (gettimeofday);

use ISC::CC::Group::Connect;

use ISC::SRS::EPPConnection;
use ISC::SRS::EPPRegistrar;
use ISC::SRS::EPPResultCode;

use ISC::Random;

my $rand = new ISC::Random(strength => "weak");

# random domain name to register
my $dname = $rand->hex(8);
$dname =~ tr/0123456789/MNOPQRSTUV/;

# random host name to register
my $hname = $rand->hex(8);
$hname =~ tr/0123456789/MNOPQRSTUV/;

my $commands = [ { cmd => 'domain_check',
		   req => { names => [ 'ISC.ORG',
				       'FOO.COM',
				       'FOO.ORG.ISC.ORG' ] } },

		 { cmd => 'domain_check',
		   req => { names => [ 'ORG.ISC.ORG '] } },

		 { cmd => 'host_check',
		   req => { names => [ 'NS1.TLD.ISC.ORG',
				       'NS.FOO.COM',
				       'AUTH00.NS.UU.NET' ] } },

		 { cmd => 'host_info',
		   req => { name => 'NS1.TLD.ISC.ORG' } },

		 { cmd => 'contact_check',
		   req => { ids => [ 'sh8013',
				     'c1',
				     'sah8013',
				     '8013sah' ] } },

		 { cmd => 'contact_info',
		   req => { id => "c1" } },

		 { cmd => 'contact_info',
		   req => { id => "c2" } },

		 { cmd => 'contact_info',
		   req => { id => "notfound" },
		   expect => EPP_RF_NOTEXISTS },

		 { cmd => 'host_add',
		   req => { name => $hname . ".FOO.ORG.ISC.org",
			    addrs => [ [ 4, '192.168.1.10' ],
				       [ 4, '192.168.2.10' ],
				       [ 6, 'fec0:0:0:bb:0290:27ff:1111:0001', ],
				       [ 6, 'fec0:0:0:bb:0290:27ff:1112:0001' ] ] },
		   expect => EPP_RF_DATAMGT },

		 { cmd => 'host_add',
		   req => { name => $hname . ".NS.UU.NET",
			    addrs => [ [ 4, '192.168.1.10' ],
				       [ 4, '192.168.2.10' ],
				       [ 6, 'fec0:0:0:bb:0290:27ff:1111:0001' ],
				       [ 6, 'fec0:0:0:bb:0290:27ff:1112:0001' ] ] },
		   expect => EPP_RF_PARAM },

		 { cmd => 'host_add',
		   req => { name => $hname . ".NS.UU.NET",
			    status => [ [ 'clientDeleteProhibited', 'en', 'foo' ] ] } },

		 { cmd => 'host_add',
		   req => { name => $hname . ".TEST.ORG.ISC.org",
			    status => [ [ 'clientDeleteProhibited', 'en', 'notouchy' ] ],
			    addrs => [ [ 4, '192.168.1.10' ],
				       [ 4, '192.168.2.10' ],
				       [ 6, 'fec0:0:0:bb:0290:27ff:1111:0001' ],
				       [ 6, 'fec0:0:0:bb:0290:27ff:1112:0001' ] ] } },

		 { cmd => 'host_mod',
		   req => { name => $hname . ".TEST.ORG.ISC.org",
			    addr_del => [ [ 6, 'fec0:0:0:bb:0290:27ff:1111:0001' ] ],
			    addr_add => [ [ 4, "1.1.2.2" ] ],
			    status_del => [ 'clientDeleteProhibited' ],
			    status_add => [ [ 'clientUpdateProhibited' ],
					    [ 'clientDeleteProhibited', 'en', '!change' ] ] } },

		 { cmd => 'host_info',
		   req => { name => $hname . ".TEST.ORG.ISC.org" } },

		 { cmd => 'domain_info',
		   req => { name => "ORG.ISC.org" } },

		 { cmd => 'domain_add',
		   req => { name => $dname . ".ORG.ISC.org",
			    nameservers => [ 'NS1.TLD.ISC.ORG',
					     'NS2.TLD.ISC.ORG' ],
			    contacts => [ [ 'admin', 'C1' ],
					  [ 'tech', 'C1' ] ] } },

		 { cmd => 'domain_info',
		   req => { name => $dname . ".ORG.ISC.org" } },


		 { cmd => 'domain_add',
		   req => { name => ++$dname . ".ORG.ISC.org",
			    owner_id => 'C2',
			    authinfo => { type => "pw", _val => "grr" },
			    nameservers => [ 'NS1.TLD.ISC.ORG',
					     'NS2.TLD.ISC.ORG' ],
			    contacts => [ [ 'admin', 'C1' ],
					  [ 'tech', 'C1' ] ],
			    status => [ [ 'clientHold', 'en', 'Bad customer' ],
					[ 'clientTransferProhibited' ] ] } },

		 { cmd => 'domain_info',
		   req => { name => $dname . ".ORG.ISC.org" } },

		 { cmd => 'domain_mod',
		   req => { name => $dname . ".ORG.isc.org",
			    owner_id => 'C1',
			    nameservers_del => [ 'NS1.TLD.ISC.ORG' ],
			    nameservers_add => [ 'NS3.TLD.ISC.ORG' ],
			    contacts_add => [ [ 'admin', 'C2' ] ],
			    contacts_del => [ [ 'admin', 'C1' ] ],
			    status_add => [ [ 'clientRenewProhibited' ] ],
			    status_del => [ 'clientTransferProhibited' ],
			    authinfo => { type => "pw", _val => "grr2" } } },

		 { cmd => 'domain_info',
		   req => { name => $dname . ".ORG.ISC.org" } },

		 { cmd => 'domain_del',
		   req => { name => $dname . ".ORG.ISC.org" } },

		 { cmd => 'domain_info',
		   req => { name => $dname . ".ORG.ISC.org" },
		   expect => EPP_RF_NOTEXISTS },

		 { cmd => 'domain_del',
		   req => { name => $dname . ".ORG.ISC.orgx" },
		   expect => EPP_RF_NOTEXISTS },

		 { cmd => 'contact_del',
		   req => { id => "c1" },
		   expect => EPP_RF_INUSE },

		 { cmd => 'contact_add',
		   req => { id => 'soon-to-be-deleted',
			    voice => '+1.3015551212',
			    voice_ext => '12345',
			    fax => '+1.3015551213',
			    email => 'nobody@example.com',
			    authinfo => { type => "pw", _val => "nobody-pw" },
			    postal => {
				"loc" =>
				{ org => "Address Organization #1",
				  name => "Emilio Lizardo",
				  street => [
				  	"10101 EIEIO Drive",
				  	"Insanity Asylum",
				  	"street3 line",
				  ],
				  city => "Rockville",
				  sp => "MD",
				  pc => "20850-0000",
				  cc => "US",
				},
				"int" =>
				{ org => "Address Organization #2",
				  name => "John Whorfin",
				  street => [
					"12345 YoyoDyne Way",
				  	"Mailstop: The PITT",
				  ],
				  city => "Grover's Mill",
				  sp => "NJ",
				  pc => "08690-0000",
				  cc => "US",
				}
			    },
			    status => [ [ 'clientDeleteProhibited', 'en', 'no whack' ],
					[ 'clientTransferProhibited' ] ] } },

		 { cmd => 'contact_info',
		   req => { id => "soon-to-be-deleted" } },

		 { cmd => 'contact_del',
		   req => { id => "soon-to-be-deleted" },
		   expect => EPP_RF_DATAMGT },

		 { cmd => 'contact_mod',
		   req => { id => "soon-to-be-deleted",
			    status_add => [ [ 'clientUpdateProhibited', 'de', 'Achtung! Kein Veraenderungen' ] ],
			    status_del => [ 'clientDeleteProhibited' ],
			    voice => "+1.3015554321",
			    voice_ext => "",
			    fax => "" } },

		 { cmd => 'contact_info',
		   req => { id => "soon-to-be-deleted" } },

		 { cmd => 'contact_mod',
		   req => { id => "soon-to-be-deleted",
			    status_del => [ 'clientUpdateProhibited' ] } },

		 { cmd => 'contact_mod',
		   req => { id => "soon-to-be-deleted",
			    postal_del => { loc => {} } } },

		 { cmd => 'contact_mod',
		   req => { id => "soon-to-be-deleted",
			    postal_chg => { int => {
				org => 'YoyoDyne Propulsion Systems',
				street => [ '1 YoyoDyne Way' ],
				city => 'Groovy Mills' } } } },

		 { cmd => 'contact_mod',
		   req => { id => "soon-to-be-deleted",
			    postal_chg => { loc => {
				org => 'YoyoDyne Propulsion Systems',
				street => [ '1 YoyoDyne Way' ],
				city => 'Groovy Mills' } } },
		   expect => EPP_RF_NOTEXISTS },

		 { cmd => 'contact_info',
		   req => { id => "soon-to-be-deleted" } },

		 { cmd => 'contact_del',
		   req => { id => "soon-to-be-deleted" } },

		 { cmd => 'logout', req => {}, expect => EPP_RS_END }
];

my $epp;

my $sock = new ISC::CC::Group::Connect(connect_cb => \&_connect_cb,
				       connect_error_cb => \&_connect_err,
				       error_cb => \&_msg_err,
				       timeout => 10);


Event->timer(cb => \&_tick, interval => 1);

my $transactions = 0;
my $start = gettimeofday;

sub _tick {
    my ($ev) = @_;

    my $now = gettimeofday;
    my $interval = $now - $start;
    $start = $now;

    if ($transactions) {
	printf("%04d done in %8.3f seconds (%8.4f/sec)\n",
	       $transactions, $interval, $transactions / $interval);
	$transactions = 0;
    }

    $sock->get_stats;
}

sub _connect_cb {
    my ($g, $sock) = @_;

    warn "connected, local name is " . $g->myname;

    $epp = new ISC::SRS::EPPConnection(msgbus => $g,
				       sid => "test-xaction.$$");

    $epp->login(cb => \&login_cb,
		data => 'data',
		cid => 'testclient',
		req => {
		    id => 'isc',
		    password => 'isc_pw',
		});
}

sub _connect_err {
    my ($foo, $msg) = @_;

    warn "ERROR: $msg";
    unloop();
}

sub _msg_err {
    my ($lp, $msg) = @_;

    print "Error on socket: " . $lp->name . ": $msg\n";
    unloop();
}

sub _group_foo_cb {
    my ($c, $msg, $wmsg) = @_;

    $transactions++;

    if (!defined($msg)) {
	die "timeout!";
    }

    print Data::Dumper->Dump([$wmsg, $msg], ['wmsg', 'msg']);
}

sub login_cb {
    my ($epp, $data, $result, $app) = @_;

    my $rs = $app->{registrar};
    if ($rs) {
	delete $app->{registrar};
	print "Got registrar_id " . $rs->id . "\n";
    }

    print Data::Dumper->Dump([$result, $app], ['result', 'app']);

    die "no registrar_id returned" unless ($rs);

    send_next_cmd($rs);
}

sub generic_cb {
    my ($rs, $data, $result, $app) = @_;
    my $cmd = $data;

    print '-' x 70, "\nResponse to:  $cmd->{cmd}\n";
    print Data::Dumper->Dump([$result, $app],
			     ['result', 'app']);

    my $expect = $cmd->{expect};
    $expect = EPP_RS_SUCCESS unless (defined($expect));
    if ($expect != $result->{resultcode}) {
	print "***** Expected $expect, got $result->{resultcode}\n";
    }

    send_next_cmd($rs);
}

sub send_next_cmd {
    my ($rs) = @_;

    my $cmd = shift @$commands;
    if (!$cmd) {
	unloop();
	return;
    }

    my $cmdname = $cmd->{cmd};

    $rs->$cmdname(data => $cmd,
		  req => $cmd->{req},
		  cb => \&generic_cb);
}

$SIG{PIPE} = "IGNORE";

$Event::DIED = sub {
    Event::verbose_exception_handler(@_);
    Event::unloop_all();
};

Event::loop();
