package kea

import (
	"context"
	log "github.com/sirupsen/logrus"
	"github.com/pkg/errors"

	"isc.org/stork/server/agentcomm"
	"isc.org/stork/server/database/model"
	"isc.org/stork/util"
	"time"
)

// Represents the status of the local server (the one that
// responded to the command).
type HALocalStatus struct {
	Role   string
	Scopes []string
	State  string
}

// Represents the status of the remote server.
type HARemoteStatus struct {
	Age        int64
	InTouch    bool     `json:"in-touch"`
	Role       string
	LastScopes []string `json:"last-scopes"`
	LastState  string   `json:"last-state"`
}

// Represents the status of the HA enabled Kea servers.
type HAServersStatus struct {
	Local  HALocalStatus
	Remote HARemoteStatus
}

// Represents a response from the single Kea server to the status-get
// command. The HAServers value is nil if it is not present in the
// response.
type Status struct {
	Pid               int64
	Uptime            int64
	Reload            int64
	HAServers         *HAServersStatus `json:"ha-servers"`
	Daemon            string
}

type AppStatus []Status

// Sends the status-get command to Kea DHCP servers and returns this status to the caller.
func GetDHCPStatus(ctx context.Context, agents agentcomm.ConnectedAgents, dbApp *dbmodel.App) (AppStatus, error) {
	// This command is only sent to the DHCP deamons.
	daemons, _ := agentcomm.NewKeaDaemons(dbApp.GetActiveDHCPDeamonNames()...)

	// It takes no arguments, thus the last parameter is nil.
	cmd, _ := agentcomm.NewKeaCommand("status-get", daemons, nil)

	// todo: hardcoding 2s timeout is a temporary solution. We need better
	// control over the timeouts.
	ctx, cancel := context.WithTimeout(ctx, 2*time.Second)
	defer cancel()

	url := storkutil.HostWithPortUrl(dbApp.CtrlAddress, dbApp.CtrlPort)

	// The Kea response will be stored in this slice of structures.
	response := []struct {
		agentcomm.KeaResponseHeader
		Arguments *Status
	}{}

	// Send the command and receive the response.
	err := agents.ForwardToKeaOverHttp(ctx, url, dbApp.Machine.Address, dbApp.Machine.AgentPort, cmd, &response)
	if err != nil {
		return nil, err
	}

	// Extract the status value.
	appStatus := AppStatus{}
	for _, r := range response {
		if r.Result != 0 && (len(r.Daemon) > 0) {
			log.Warn(errors.Errorf("status-get command failed for Kea daemon %s", r.Daemon))

		} else if r.Arguments != nil {
			appStatus = append(appStatus, *r.Arguments)
			appStatus[len(appStatus)-1].Daemon = r.Daemon
		}
	}

	return appStatus, nil
}

