/////////////////////////////////////////////////////////////
//
// pgAdmin 4 - PostgreSQL Tools
//
// Copyright (C) 2013 - 2022, The pgAdmin Development Team
// This software is released under the PostgreSQL Licence
//
//////////////////////////////////////////////////////////////

import { getPanelView } from './panel_view';

define(
  ['underscore', 'sources/pgadmin', 'jquery', 'wcdocker'],
  function(_, pgAdmin, $) {

    var pgBrowser = pgAdmin.Browser = pgAdmin.Browser || {},
      wcDocker = window.wcDocker;

    pgAdmin.Browser.Panel = function(options) {
      var defaults = [
        'name', 'title', 'width', 'height', 'showTitle', 'isCloseable',
        'isPrivate', 'isLayoutMember', 'content', 'icon', 'events', 'onCreate', 'elContainer',
        'canHide', 'limit', 'extraClasses', 'canMaximise',
      ];
      _.extend(this, _.pick(options, defaults));
    };

    _.extend(pgAdmin.Browser.Panel.prototype, {
      name: '',
      title: '',
      width: 300,
      height: 600,
      showTitle: true,
      isCloseable: true,
      isPrivate: false,
      isLayoutMember: true,
      content: '',
      icon: '',
      panel: null,
      onCreate: null,
      elContainer: false,
      canMaximise: false,
      limit: null,
      extraClasses: null,
      load: function(docker, title) {
        var that = this;
        if (!that.panel) {
          docker.registerPanelType(that.name, {
            title: that.title,
            isPrivate: that.isPrivate,
            limit: that.limit,
            isLayoutMember: that.isLayoutMember,
            onCreate: function(myPanel) {
              $(myPanel).data('pgAdminName', that.name);
              myPanel.initSize(that.width, that.height);

              if (!that.showTitle)
                myPanel.title(false);
              else {
                var title_elem = '<a href="#" tabindex="-1" class="panel-link-heading">' + (title || that.title) + '</a>';
                myPanel.title(title_elem);
                if (that.icon != '')
                  myPanel.icon(that.icon);
              }

              var $container = $('<div>', {
                'class': 'pg-panel-content',
              }).append($(that.content));

              // Add extra classes
              if (!_.isNull('extraClasses')) {
                $container.addClass(that.extraClasses);
              }

              myPanel.maximisable(!!that.canMaximise);
              myPanel.closeable(!!that.isCloseable);
              myPanel.layout().addItem($container);
              that.panel = myPanel;
              if (that.events && _.isObject(that.events)) {
                _.each(that.events, function(v, k) {
                  if (v && _.isFunction(v)) {
                    myPanel.on(k, v);
                  }
                });
              }
              _.each([
                wcDocker.EVENT.UPDATED, wcDocker.EVENT.VISIBILITY_CHANGED,
                wcDocker.EVENT.BEGIN_DOCK, wcDocker.EVENT.END_DOCK,
                wcDocker.EVENT.GAIN_FOCUS, wcDocker.EVENT.LOST_FOCUS,
                wcDocker.EVENT.CLOSED, wcDocker.EVENT.BUTTON,
                wcDocker.EVENT.ATTACHED, wcDocker.EVENT.DETACHED,
                wcDocker.EVENT.MOVE_STARTED, wcDocker.EVENT.MOVE_ENDED,
                wcDocker.EVENT.MOVED, wcDocker.EVENT.RESIZE_STARTED,
                wcDocker.EVENT.RESIZE_ENDED, wcDocker.EVENT.RESIZED,
                wcDocker.EVENT.SCROLLED,
              ], function(ev) {
                myPanel.on(ev, that.eventFunc.bind(myPanel, ev));
              });

              if (that.onCreate && _.isFunction(that.onCreate)) {
                that.onCreate.apply(that, [myPanel, $container]);
              }

              // Prevent browser from opening the drag file.
              // Using addEventListener to avoid conflict with jquery.drag
              ['dragover', 'drop'].forEach((eventName)=>{
                $container[0].addEventListener(eventName, function(event) {
                  event.stopPropagation();
                  event.preventDefault();
                });
              });

              if (that.elContainer) {
                myPanel.pgElContainer = $container;
                $container.addClass('pg-el-container');
                _.each([
                  wcDocker.EVENT.RESIZED, wcDocker.EVENT.ATTACHED,
                  wcDocker.EVENT.DETACHED, wcDocker.EVENT.VISIBILITY_CHANGED,
                ], function(ev) {
                  myPanel.on(ev, that.resizedContainer.bind(myPanel));
                });
                that.resizedContainer.apply(myPanel);
              }

              if (myPanel._type == 'dashboard') {
                getPanelView(
                  pgBrowser.tree,
                  $container[0],
                  pgBrowser,
                  myPanel._type
                );
              }

              // Re-render the dashboard panel when preference value 'show graph' gets changed.
              pgBrowser.onPreferencesChange('dashboards', function() {
                getPanelView(
                  pgBrowser.tree,
                  $container[0],
                  pgBrowser,
                  myPanel._type
                );
              });

              // Re-render the dashboard panel when preference value gets changed.
              pgBrowser.onPreferencesChange('graphs', function() {
                getPanelView(
                  pgBrowser.tree,
                  $container[0],
                  pgBrowser,
                  myPanel._type
                );
              });

              _.each([wcDocker.EVENT.CLOSED, wcDocker.EVENT.VISIBILITY_CHANGED],
                function(ev) {
                  myPanel.on(ev, that.handleVisibility.bind(myPanel, ev));
                });

              pgBrowser.Events.on('pgadmin-browser:tree:selected', () => {

                if(myPanel.isVisible() && myPanel._type !== 'properties') {
                  getPanelView(
                    pgBrowser.tree,
                    $container[0],
                    pgBrowser,
                    myPanel._type
                  );
                }
              });

              pgBrowser.Events.on('pgadmin:database:connected', () => {

                if(myPanel.isVisible() && myPanel._type !== 'properties') {
                  getPanelView(
                    pgBrowser.tree,
                    $container[0],
                    pgBrowser,
                    myPanel._type
                  );
                }
              });

              pgBrowser.Events.on('pgadmin-browser:tree:refreshing', () => {

                if(myPanel.isVisible() && myPanel._type !== 'properties') {
                  getPanelView(
                    pgBrowser.tree,
                    $container[0],
                    pgBrowser,
                    myPanel._type
                  );
                }
              });
            },
          });
        }
      },
      eventFunc: function(eventName) {
        var name = $(this).data('pgAdminName');
        try {
          pgBrowser.Events.trigger(
            'pgadmin-browser:panel', eventName, this, arguments
          );
          pgBrowser.Events.trigger(
            'pgadmin-browser:panel:' + eventName, this, arguments
          );

          if (name) {
            pgBrowser.Events.trigger(
              'pgadmin-browser:panel-' + name, eventName, this, arguments
            );
            pgBrowser.Events.trigger(
              'pgadmin-browser:panel-' + name + ':' + eventName, this, arguments
            );
          }
        } catch (e) {
          console.warn(e.stack || e);
        }
      },
      resizedContainer: function() {
        var p = this;

        if (p.pgElContainer && !p.pgResizeTimeout) {
          if (!p.isVisible()) {
            clearTimeout(p.pgResizeTimeout);
            p.pgResizeTimeout = null;

            return;
          }
          p.pgResizeTimeout = setTimeout(
            function() {
              var w = p.width(),
                elAttr = 'xs';
              p.pgResizeTimeout = null;

              /** Calculations based on https://getbootstrap.com/docs/4.1/layout/grid/#grid-options **/
              if (w >= 480) {
                elAttr = 'sm';
              }
              if (w >= 768) {
                elAttr = 'md';
              }
              if (w >= 992) {
                elAttr = 'lg';
              }
              if (w >=1200) {
                elAttr = 'xl';
              }

              p.pgElContainer.attr('el', elAttr);
            },
            100
          );
        }
      },
      handleVisibility: function(eventName) {
        if (_.isNull(pgBrowser.tree)) return;

        let selectedPanel = pgBrowser.docker.findPanels(this._type)[0];
        let isPanelVisible = selectedPanel.isVisible();
        var $container = selectedPanel
          .layout()
          .scene()
          .find('.pg-panel-content');

        if (isPanelVisible && selectedPanel._type !== 'properties') {
          if (eventName == 'panelVisibilityChanged' && selectedPanel._type !== 'properties') {
            getPanelView(
              pgBrowser.tree,
              $container[0],
              pgBrowser,
              this._type
            );
          }
        }
        if (eventName == 'panelClosed' && selectedPanel._type == 'dashboard') {
          getPanelView(
            pgBrowser.tree,
            $container[0],
            pgBrowser,
            this._type,
            false
          );
        }
      }

    });

    return pgAdmin.Browser.Panel;
  });
