/////////////////////////////////////////////////////////////
//
// pgAdmin 4 - PostgreSQL Tools
//
// Copyright (C) 2013 - 2022, The pgAdmin Development Team
// This software is released under the PostgreSQL Licence
//
//////////////////////////////////////////////////////////////

import _ from 'lodash';
import React, { useEffect } from 'react';
import PgTable from 'sources/components/PgTable';
import gettext from 'sources/gettext';
import PropTypes from 'prop-types';
import Notify from '../../../../static/js/helpers/Notifier';
import getApiInstance from 'sources/api_instance';
import { makeStyles } from '@material-ui/core/styles';
import sizePrettify from 'sources/size_prettify';
import { getURL } from '../../../static/utils/utils';
import Loader from 'sources/components/Loader';
import EmptyPanelMessage from '../../../../static/js/components/EmptyPanelMessage';
const useStyles = makeStyles((theme) => ({
  emptyPanel: {
    minHeight: '100%',
    minWidth: '100%',
    background: theme.otherVars.emptySpaceBg,
    overflow: 'auto',
    padding: '8px',
    display: 'flex',
  },
  panelIcon: {
    width: '80%',
    margin: '0 auto',
    marginTop: '25px !important',
    position: 'relative',
    textAlign: 'center',
  },
  panelMessage: {
    marginLeft: '0.5rem',
    fontSize: '0.875rem',
  },
  autoResizer: {
    height: '100% !important',
    width: '100% !important',
    background: theme.palette.grey[400],
    padding: '7.5px',
    overflowX: 'auto !important',
    overflowY: 'hidden !important',
    minHeight: '100%',
    minWidth: '100%',
  },
}));

function getColumn(data, singleLineStatistics) {
  let columns = [], column;
  if (!singleLineStatistics) {
    if (!_.isUndefined(data)) {
      data.forEach((row) => {
        if (row.name == 'Size') {
          column = {
            Header: row.name,
            accessor: row.name,
            sortble: true,
            resizable: true,
            disableGlobalFilter: false,
            sortType: ((rowA, rowB, id) => {
              let val1 = rowA.values[id];
              let val2 = rowB.values[id];
              const sizes = ['bytes', 'kB', 'MB', 'GB', 'TB', 'PB', 'EB', 'ZB', 'YB'];
              sizes.some((t, i) => {
                if (!_.isNull(rowA.values[id]) && typeof (rowA.values[id]) == 'string' && rowA.values[id].indexOf(t) > -1) {
                  val1 = (parseInt(rowA.values[id]) * Math.pow(1024, i));
                }

                if (!_.isNull(rowB.values[id]) && typeof (rowB.values[id]) == 'string' && rowB.values[id].indexOf(t) > -1) {
                  val2 = parseInt(rowB.values[id]) * Math.pow(1024, i);
                }

              });

              if ((val1) > (val2) || _.isNull(val2)) {
                return 1;
              }
              if ((val2) > (val1) || _.isNull(val1)) {
                return -1;
              }
              return 0;

            })
          };
        }else{
          column = {
            Header: row.name,
            accessor: row.name,
            sortble: true,
            resizable: true,
            disableGlobalFilter: false,
          };

        }
        columns.push(column);
      });
    }
    return columns;
  } else {
    columns = [
      {
        Header: gettext('Statistics'),
        accessor: 'name',
        sortble: true,
        resizable: true,
        disableGlobalFilter: false,
      },
      {
        Header: 'Value',
        accessor: 'value',
        sortble: true,
        resizable: true,
        disableGlobalFilter: false,
      },
    ];
  }
  return columns;
}

function getTableData(res, node) {
  let nodeStats = [],
    colData;
  if (res.data.data) {
    let data = res.data.data;
    if (node.hasCollectiveStatistics || data['rows'].length > 1) {
      data.rows.forEach((row) => {
        // Prettify the field values
        if (!_.isEmpty(node.statsPrettifyFields)) {
          node.statsPrettifyFields.forEach((field) => {
            row[field] = sizePrettify(row[field]);
          });
        }
        nodeStats.push({ ...row, icon: '' });
      });
      colData = getColumn(data.columns, false);
    } else {
      nodeStats = createSingleLineStatistics(data, node.statsPrettifyFields);
      colData = getColumn(data.columns, true);
    }
  }
  return [nodeStats, colData];
}

function createSingleLineStatistics(data, prettifyFields) {
  var row = data['rows'][0],
    columns = data['columns'],
    res = [],
    name,
    value;

  for (var idx in columns) {
    name = columns[idx]['name'];
    if (row && row[name]) {
      value =
        _.indexOf(prettifyFields, name) != -1
          ? sizePrettify(row[name])
          : row[name];
    } else {
      value = null;
    }

    res.push({
      name: name,
      value: value,
      icon: '',
    });
  }

  return res;
}

export default function Statistics({ nodeData, item, node, ...props }) {
  const classes = useStyles();
  const [tableData, setTableData] = React.useState([]);

  const [msg, setMsg] = React.useState('');
  const [loaderText, setLoaderText] = React.useState('');
  const [columns, setColumns] = React.useState([
    {
      Header: 'Statictics',
      accessor: 'name',
      sortble: true,
      resizable: true,
      disableGlobalFilter: false,
    },
    {
      Header: 'Value',
      accessor: 'value',
      sortble: true,
      resizable: true,
      disableGlobalFilter: false,
    },
  ]);

  useEffect(() => {
    let url,
      message = gettext('Please select an object in the tree view.');

    if (node) {
      url = getURL(nodeData, true, props.treeNodeInfo, node, item, 'stats');

      message = gettext('No statistics are available for the selected object.');

      const api = getApiInstance();
      if (node.hasStatistics) {
        setLoaderText('Loading...');
        api({
          url: url,
          type: 'GET',
        })
          .then((res) => {
            let [nodeStats, colData] = getTableData(res, node);
            setTableData(nodeStats);
            if (!_.isUndefined(colData)) {
              setColumns(colData);
            }
            setLoaderText('');
          })
          .catch((err) => {
            // show failed message.
            setLoaderText('');

            if (err?.response?.data?.info == 'CRYPTKEY_MISSING') {
              Notify.pgNotifier('error', err.request, 'The master password is not set', function(msg) {
                setTimeout(function() {
                  if (msg == 'CRYPTKEY_SET') {
                    setMsg('No statistics are available for the selected object.');
                  } else if (msg == 'CRYPTKEY_NOT_SET') {
                    setMsg(gettext('The master password is not set.'));
                  }
                }, 100);
              });
            } else {
              Notify.alert(
                gettext('Failed to retrieve data from the server.'),
                gettext(err.message)
              );
              setMsg(gettext('Failed to retrieve data from the server.'));
            }
          });
      } else {
        setLoaderText('');
        setMsg('No statistics are available for the selected object.');
      }
    }
    if (message != '') {
      setMsg(message);
    }
    return () => {
      setTableData([]);
    };
  }, [nodeData]);

  return (
    <>
      {tableData.length > 0 ? (
        <PgTable
          className={classes.autoResizer}
          columns={columns}
          data={tableData}
          msg={msg}
          type={'panel'}
        ></PgTable>
      ) : (
        <div className={classes.emptyPanel}>
          <Loader message={loaderText} />
          <EmptyPanelMessage text={gettext(msg)}/>
        </div>
      )}
    </>
  );
}

Statistics.propTypes = {
  res: PropTypes.array,
  nodeData: PropTypes.object,
  item: PropTypes.object,
  treeNodeInfo: PropTypes.object,
  node: PropTypes.func,
};
