/*
    Library: PostgreSQL binding for Objective-C
    Copyright (C) 2003 Alexander V. Diemand

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public
    License as published by the Free Software Foundation; either
    version 2.1 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with this library; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#include "PostgreSQL.oh"


@implementation PostgreSQL : NSObject
{
	@private
	PGconn *conn;
	PGresult *result;
}


+(PostgreSQL*)connect
{
	id db,host,port;
	char *str = getenv("PGDATABASE");
	if (!str)
	{
		NSLog(@"you must set $PGDATABASE.");
		return nil;
	}
	db = [NSString stringWithCString: str];
	str = getenv("PGHOST");
	if (!str)
	{
		host = @"localhost";
	} else {
		host = [NSString stringWithCString: str];
	}
	str = getenv("PGPORT");
	if (!str)
	{
		port = @"5432";
	} else {
		port = [NSString stringWithCString: str];
	}
	PostgreSQL *pq = [[PostgreSQL alloc]init];
//	NSLog(@"connectTo: %@ on: %@ at: %@\n",db,host,port);

        pq->conn = PQsetdb([host cString], [port cString], NULL, NULL, [db cString]);
        if (PQstatus(pq->conn) == CONNECTION_BAD)
        {
		NSLog(@"OBJCPQ: Connection to database '%@' failed.\n%s", db, PQerrorMessage(pq->conn));
		pq->conn = NULL;
		return nil;
	}
	return [pq autorelease];
}


+(PostgreSQL*)connectTo: (NSString*)db on: (NSString*)host at: (NSString*)port
{
	PostgreSQL *pq = [[PostgreSQL alloc]init];
//	NSLog(@"connectTo: %@ on: %@ at: %@\n",db,host,port);

        pq->conn = PQsetdb([host cString], [port cString], NULL, NULL, [db cString]);
        if (PQstatus(pq->conn) == CONNECTION_BAD)
        {
		NSLog(@"OBJCPQ: Connection to database '%@' failed.\n%s", db, PQerrorMessage(pq->conn));
		pq->conn = NULL;
		return nil;
	}
	return [pq autorelease];
}

-(id)init
{
	[super init];
	conn = NULL;
	result = NULL;
	return self;
}


-(void)dealloc
{
	if (conn != NULL)
	{
		PQfinish(conn);
	}
	if (result != NULL)
	{
	        PQclear(result);
	}
	[super dealloc];
}


-(id)disconnect
{
	if (conn == NULL)
	{
		return nil;
	}
	if (result != NULL)
	{
	        PQclear(result);
	}
	PQfinish(conn);
	conn = NULL;
	result = NULL;
	return self;
}


-(BOOL)query1: (NSString*)q
{
	if (conn == NULL)
	{
		return NO;
	}
	if (result != NULL)
	{ /* there is already a result set */
                PQclear(result);
		result = NULL;
	}
	result = PQexec (conn, [q cString]);
        if (PQresultStatus(result) != PGRES_COMMAND_OK)
	{
                PQclear(result);
		result = NULL;
                return NO;
        }
	PQclear(result);
	result = NULL;
	return YES;
}


-(BOOL)query: (NSString*)q
{
	if (conn == NULL)
	{
		return NO;
	}
	if (result != NULL)
	{ /* there is already a result set */
		PQclear(result);
	}

	result = PQexec (conn, [q cString]);
	int resulttype = PQresultStatus(result);
	if ((resulttype != PGRES_EMPTY_QUERY) &&
	    (resulttype != PGRES_COMMAND_OK) &&
	    (resulttype != PGRES_TUPLES_OK))
	{
		NSLog(@"OBJCPQ: Error %s on query: %s\n",PQresStatus(resulttype),PQresultErrorMessage(result));
		PQclear(result);
		result = NULL;
		return NO;
	}
	return YES;
}


-(unsigned long)rows
{
	if (result == NULL)
	{
		return 0L;
	}
	return PQntuples(result);
}


-(unsigned long)fields
{
	if (result == NULL)
	{
		return 0L;
	}
	return PQnfields(result);
}


/* just used locally here */
#define INT2OID         21
#define INT4OID         23
#define OIDOID          26
#define FLOAT4OID       700
#define FLOAT8OID       701
#define CASHOID         790

/* 0 >= row < allrows */
-(NSArray*)getRow: (unsigned long)row
{
	id res;
	if (conn == NULL || result == NULL)
	{
		NSLog(@"OBJCPQ: no result available.");
		return nil;
	}
	unsigned long allrows = [self rows];
	unsigned long allfields = [self fields];
	if (row >= allrows)
	{
		NSLog(@"OBJCPQ: no such row in result set: %d.",row);
		return nil;
	}
	res = [NSMutableArray arrayWithCapacity: (unsigned int)allfields];
	id obj;
	char *tchar;
	int fnum=0;
	for (fnum=0; fnum<[self fields]; fnum++) {
		tchar=PQgetvalue(result,row,fnum);
		switch(PQftype(result,fnum)) {
		case INT2OID:
		case INT4OID:
		case OIDOID:
		{ /* integer types */
			long tval = atol(tchar);
			obj = [NSDecimalNumber numberWithLong: tval];
			break;
		}
		case FLOAT4OID:
		case FLOAT8OID:
		case CASHOID:
		{ /* floating point numbers */
			double tval = atof(tchar);
			obj = [NSDecimalNumber numberWithDouble: tval];
			break;
		}
		default:
		{ /* everything else is just a string */
			obj = [NSString stringWithCString: tchar];
			break;
		}
		} /* switch over field types */
		[res addObject: obj];
	} /* for each field */
	return res;
}


@end

