namespace eval ::commandline {
	namespace export GetArgv0 GetOptions
}


######################################################################
##GetArgv0
##  Description:
##    Returns the portion of the filename without the path and 
##    without the suffix
######################################################################
proc ::commandline::GetArgv0 {} {
    global argv0

    return [file rootname [file tail $argv0]]
}

######################################################################
##GetOptC:
##  Description:
##    Similar to standard C getopt.  Given an list of options
##    and an array of arguments, this function will parse the
##    first argument.
##  Inputs:
##    pArgv 
##      The name of the argument list to parse
##      WARNING:  This string is MODIFIED!  Upon successful completion, 
##      this string will hold the REMAINDER of arguments left to parse!
##    pOptions
##      A list of acceptable arguments.  If an option in the list ends
##      in ".arg", i.e. "i.arg", the function will require an argument
##      to follow the option, i.e. "-i test" is okay, but "-i" by itself
##      will return an error.  
##    pOption
##      Contains the option that was parsed, without the "-"
##    pValue
##      Contains the value of the parsed option, this will be 1, or
##      if ".arg" was part of the option name, then it will contain
##      the value from the command line that immediately followed the
##      option
##  Output
##    1    A valid option was located
##    0    No more options left to parse
##   -1    Not a valid argument
##   -2    Argument is missing a value
######################################################################
proc ::commandline::GetOptC {pArgv pOptions pOption pValue} {
	upvar 1 $pArgv   uArgv
	upvar 1 $pOption uOption
	upvar 1 $pValue  uValue

    set uValue ""
    set uOption ""
    
    set lResult 0

    if {([llength $uArgv] != 0)} {
		switch -glob -- [set lArgument [lindex $uArgv 0]] {
			"--" {
				set uArgv [lrange $uArgv 1 end]
			}
			"-*" {
				set uOption [string range $lArgument 1 end]
				if {([lsearch -exact $pOptions $uOption] != -1)} {
					set uValue 1
					set lResult 1
					set uArgv [lrange $uArgv 1 end]
				} elseif {([lsearch -exact $pOptions "$uOption.arg"] != -1)} {
					set lResult 1
					set uArgv [lrange $uArgv 1 end]
					if {([llength $uArgv] != 0)} {
						set uValue [lindex $uArgv 0]
						set uArgv [lrange $uArgv 1 end]
					} else {
						set lValue $uOption
						set lResult -2
					}
				} elseif {([lsearch \
					-exact $pOptions "$uOption.required"] != -1)} {
					set lResult 1
					set uArgv [lrange $uArgv 1 end]
					if {([llength $uArgv] != 0)} {
						set uValue [lindex $uArgv 0]
						set uArgv [lrange $uArgv 1 end]
					} else {
						set lValue $uOption
						set lResult -2
					}					
				} else {
					set uValue $uOption
					set lResult -1
				}
			}
		}
	}
	return $lResult
}


# cmdline::usage --
#
#	Generate an error message that lists the allowed flags.
#
# Arguments:
#	optlist		As for cmdline::getoptions
#	usage		Text to include in the usage display. Defaults to
#			"options:"
#
# Results
#	A formatted usage message
proc commandline::Usage {pOptions {pUsage "usage:"}} {
	set lHelp {{help "Print this message"} {? "Print this message"}}
	set lOptions [concat $pOptions $lHelp] 
	set lHasDefault 0

	foreach lOption $lOptions {
		set lArgument [lindex $lOption 0]
		if {([regsub ".secret$" $lArgument {} lArgument] == 0)} {
			if {([regsub ".arg$" $lArgument {} lArgument] == 1)} {
				set lHasDefault 1
				set lDefault [lindex $lOption 1]
				set lComment [lindex $lOption 2]
				append lResult [format " %-15s   %-7s %s\n" \
					"-$lArgument value" $lDefault $lComment]
			} elseif {([regsub ".required$" $lArgument {} lArgument] == 1)} {
				set lHasDefault 1
				set lDefault ""
				set lComment [lindex $lOption 2]
				append lResult [format " %-15s   %-7s %s\n" \
					"-$lArgument value" $lDefault $lComment]
			} else {
				set lComment [lindex $lOption 1]
				append lResult [format " %-15s   %-7s %s\n" \
					"-$lArgument" "" $lComment]
			}
		}
	}
	
	if {($lHasDefault == 0)} {
		set lResult "[::commandline::GetArgv0] $pUsage\n$lResult"
	} else {
		set lResult "[format " %-15s Default" ""]\n$lResult"
		set lResult "[::commandline::GetArgv0] $pUsage\n$lResult"
	}
	return $lResult
}

# cmdline::getoptions --
#
#	Process a set of command line options, filling in defaults
#	for those not specified.  This also generates an error message
#	that lists the allowed flags if an incorrect flag is specified.
#
# Arguments:
#	arglistVar	The name of the argument list, typically argv
#	optlist		A list-of-lists where each element specifies an option
#			in the form:
#				flag default comment
#			If flag ends in ".arg" then the value is taken from the
#			command line. Otherwise it is a boolean and appears in
#			the result if present on the command line. If flag ends
#			in ".secret", it will not be displayed in the usage.
#	usage		Text to include in the usage display. Defaults to
#			"options:"
#
# Results
#	Name value pairs suitable for using with array set.

proc ::commandline::GetOptions {pArgv pOptions {pUsage "usage:"}} {
	upvar 1 $pArgv uArgv

	set lOptions {? help}
	
	foreach lOption $pOptions {
		set lArgument [lindex $lOption 0]
		if {([regsub ".secret$" $lArgument {} lArgument] == 0)} {
			lappend lOptions $lArgument
			if {([regsub ".arg$" $lArgument {} lArgument] == 1)} {
				set lDefault [lindex $lOption 1]
				set lResult($lArgument) $lDefault
			} elseif {([regsub ".required$" $lArgument {} lArgument] == 1)} {
				set lResult($lArgument) ""
				set lRequired($lArgument) 1
			} else {
				set lResult($lArgument) 0
			}
		}
	}

	set lArgc [llength $uArgv]
	set lArgList ""

	while {($uArgv != "")} {
		set lError [::commandline::GetOptC uArgv $lOptions lOption lValue]
		if {($lError == "0")} {
			lappend lArgList [lindex $uArgv 0]
			set uArgv [lrange $uArgv 1 end]
			continue
		}
		if {($lError < 0)} {
			puts [::commandline::Usage $pOptions $pUsage]
			error "[::commandline::GetArgv0]:  illegal option" {} \
				[list [::commandline::GetArgv0] 1]
		}
		set lResult($lOption) $lValue
		if {([info exists lRequired($lOption)])} {
			unset lRequired($lOption)
		}
	}
	if {([array size lRequired] != 0)} {
		puts [::commandline::Usage $pOptions $pUsage]
		error "[::commandline::GetArgv0]:  illegal option" {} \
			[list [::commandline::GetArgv0] 1]
	}
			
	if {([info exist lResult(?)]) || ([info exists lResult(help)])} {
		puts [::commandline::Usage $pOptions $pUsage]
		return ""
	}

	set lResult(argv) $lArgList
	return [array get lResult]
}

# cmdline::getfiles --
#
#	Given a list of file arguments from the command line, compute
#	the set of valid files.  On windows, file globbing is performed
#	on each argument.  On Unix, only file existence is tested.  If
#	a file argument produces no valid files, a warning is optionally
#	generated.
#
#	This code also uses the full path for each file.  If not
#	given it prepends [pwd] to the filename.  This ensures that
#	these files will never comflict with files in our zip file.
#
# Arguments:
#	patterns	The file patterns specified by the user.
#	quiet		If this flag is set, no warnings will be generated.
#
# Results:
#	Returns the list of files that match the input patterns.

proc ::commandline::GetFiles {pPatterns {pQuiet 0}} {
	set result {}
	foreach lPattern $pPatterns {
		set lFiles [glob -nocomplain -- $lPattern]
		if {($lFiles == "")} {
			if {($pQuiet != 0)} {
				puts stdout "warning: no files match \"$lPattern\""
			}
		} else {
			foreach lFile $lFiles {
				lappend lResult $lFile
			}
		}
	}

	set lFiles ""
	foreach lFile $lResult {
		set lFullPath [file join [pwd] $lFile]
	
		if {([file isfile $lFullPath])} {
			lappend lFiles $lFullPath
		} elseif {($pQuiet != 0)} {
			puts stdout "warning: no files match \"$lFile\""
		}
	}
	return $lFiles
}

package provide commandline 1.1
