/*
	utils.cpp

	Common utility functions missing in the standard CRT API.

	Project: pgjobs
	Author: Zlatko Michailov
	Created:  6-Oct-2003
	Updated: 14-Oct-2003

	This file is provided as is, with no warranty. Use at your own risk.

	Copyright (c) 2003, Zlatko Michailov

*/



//--------------------------------------------------------------------------------

#include <string.h>
#include <stdlib.h>
#include <errno.h>
#include <unistd.h>
#include <pwd.h>
#include <sys/types.h>
#include <sys/stat.h>



//--------------------------------------------------------------------------------
// Makes sure the entire path exists.
// May create more than one level of directories.
// The path may have a trailing '/'.

int mkdirhier( const char* path, int mode )
{
	const char	DirChar		= '/';
	const char	TermChar	= '\0';

	int		err			= -1;
	char*	pathBuffer	= 0;

	// Make sure the owner will have full access
	mode |= 0750;

	// We'll need to modify the path string,
	// therefore copy it to a local buffer.
	// If path is not null but empty, return error.
	errno = ENOENT;
	if ( path && *path )
	{
		errno		= ENOMEM;
		pathBuffer	= strdup( path );
	}

	// The following algorithm requires at least 1 char in pathBuffer!
	if ( pathBuffer && *pathBuffer )
	{
		bool	isDone		= false;
		int		pathLength	= strlen( pathBuffer );
		char*	end			= pathBuffer;

		// If every path component exists, the results is OK.
		// Errors may only come from mkdir().
		err		= 0;
		errno	= 0;

		// Traverse all path components
		while ( !isDone && !err )
		{
			// Find the next path component and (temporarily) terminate the buffer.
			// The last component is either at the end of the string or
			// right before a trailing '/'
			end		= strchr( end + 1, DirChar );
			isDone	= !end || ( end - pathBuffer == pathLength - 1 );
			if ( end )
			{
				*end = TermChar;
			}

			// Test the current head of the path for existance
			if ( access( pathBuffer, 0 ) )
			{
				// Since it doesn't exist, try to create it.
				// This call is the only possible source of error.
				err = mkdir( pathBuffer, mode );
			}

			// Recover the temporary buffer termination
			if ( !isDone )
			{
				*end = DirChar;
			}

		} // while ( path components )

		// Release the local path buffer
		free( pathBuffer );
	}

	return err;
}



//--------------------------------------------------------------------------------
// Re-entrant impersonation by username:
// Obtains in a re-entrant manner the uid of the user given the username.
// Sets the effective uid.
// Return:
//		 0 - success
//		-1 - error
//				ENOENT	- username not found as a user
//				EPERM	- failed to set effective uid

int setename( const char* name )
{
	passwd	pwd;
	passwd*	__dummy;
	char	buffer[ NSS_BUFLEN_PASSWD ];
	int		err;

	// Get passwd structure from the /etc/passwd config file
	err   = -!!getpwnam_r( name, &pwd, buffer, NSS_BUFLEN_PASSWD, &__dummy );
	errno = -err * ENOENT;
	if ( !err )
	{
		// Set effective uid
		err   = -!!seteuid( pwd.pw_uid );
		errno = -err * EPERM;
	}

	return err;
}



//--------------------------------------------------------------------------------

