/*
 * pg_reporter.c: entry point and xslt processor.
 *
 * Copyright (c) 2009-2010, NIPPON TELEGRAPH AND TELEPHONE CORPORATION
 */

#include "pg_reporter.h"

#include <float.h>
#include <math.h>
#include <sys/stat.h>

#include <libxml/xpathInternals.h>
#include <libxslt/extensions.h>
#include <libxslt/transform.h>
#include <libxslt/xsltutils.h>

#include "catalog/pg_type.h"
#include "mb/pg_wchar.h"

const char *PROGRAM_VERSION	= "1.0alpha1";
const char *PROGRAM_URL		= "http://pgstatsinfo.projects.postgresql.org/";
const char *PROGRAM_EMAIL	= "pgstatsinfo-general@pgfoundry.org";

#define NS_XHTML		"http://www.w3.org/1999/xhtml"
#define XSL_DEFAULT		"xsl/html.xsl"
#define VARTAG			'$'		/* XXX: consider ':' instead of $ */
#define ROWNUMOID		1

typedef struct Format
{
	char			   *name;
	xmlXPathObjectPtr (*func)(PGresult *rs, xmlNsPtr ns, const char *name);
} Format;

/* local defs */
static void pg_query(xmlXPathParserContextPtr ctxt, int nargs);
static void addNode(xmlNodePtr parent, const char *name, const char *value, Oid type);
static void toArray(StringInfo str, PGresult *rs);

static xmlXPathObjectPtr createTABLE(PGresult *rs, xmlNsPtr ns, const char *name);
static xmlXPathObjectPtr createTBODY(PGresult *rs, xmlNsPtr ns, const char *name);
static xmlXPathObjectPtr createDL(PGresult *rs, xmlNsPtr ns, const char *name);
static xmlXPathObjectPtr createULOL(PGresult *rs, xmlNsPtr ns, const char *name);
static xmlXPathObjectPtr createSPAN(PGresult *rs, xmlNsPtr ns, const char *name);
static xmlXPathObjectPtr createPRE_postgres(PGresult *rs, xmlNsPtr ns, const char *name);
static xmlXPathObjectPtr createPRE_mysql(PGresult *rs, xmlNsPtr ns, const char *name);
static xmlXPathObjectPtr createPRE_oracle(PGresult *rs, xmlNsPtr ns, const char *name);

static xmlXPathObjectPtr createCIRCLE(PGresult *rs, xmlNsPtr ns, const char *name);
static xmlXPathObjectPtr createVBAR(PGresult *rs, xmlNsPtr ns, const char *name);
static xmlXPathObjectPtr createLINE(PGresult *rs, xmlNsPtr ns, const char *name);
static xmlXPathObjectPtr createRADAR(PGresult *rs, xmlNsPtr ns, const char *name);

static const Format formats[] =
{
	{ "table", createTABLE },
	{ "tbody", createTBODY },
	{ "dl", createDL },
	{ "ul", createULOL },
	{ "ol", createULOL },
	{ "span", createSPAN},
	{ "pre-postgres", createPRE_postgres },
	{ "pre-mysql", createPRE_mysql },
	{ "pre-oracle", createPRE_oracle },
	{ "circle", createCIRCLE },
	{ "vbar", createVBAR },
	{ "line", createLINE },
	{ "radar", createRADAR },
};

static char	   *input = NULL;			/* input path */
static char	   *output = NULL;			/* output path */
static char	   *stylesheet = NULL;		/* stylesheet path */
static char	   *httpd_dir = NULL;		/* httpd dir */
static int		httpd_port = 8080;		/* httpd port */
static char	   *httpd_listen = NULL;	/* httpd listen addresses */
static char	   *init = NULL;			/* init target directory */
static char	   *parameters = NULL;		/* arguments from command line */
static List	   *pg_query_vars = NIL;	/* query variables */

/* arguments */
static struct pgut_option options[] =
{
	{ 's', 't', "template", &input },
	{ 's', 'o', "output", &output },
	{ 's', 's', "stylesheet", &stylesheet },
	{ 's', 'a', "args", &parameters },
	{ 's', 'D', "httpd-dir", &httpd_dir },
	{ 'i', 'P', "httpd-port", &httpd_port },
	{ 's', 'L', "httpd-listen", &httpd_listen },
	{ 's', 'i', "init", &init },
	{ 0 }
};

void
pgut_help(bool details)
{
	printf("%s reports a PostgreSQL database.\n\n", PROGRAM_NAME);
	printf("Usage:\n");
	printf("  %s TEMPLATE [OUTPUT]\n", PROGRAM_NAME);

	if (!details)
		return;

	printf("\nReport options:\n");
	printf("  -t, --template=TEMPLATE   template path\n");
	printf("  -o, --output=OUTPUT       output path\n");
	printf("  -s, --stylesheet=XSLT     stylesheet path\n");
	printf("  -a, --args=\"key=val&...\"  template parameters\n");
	printf("\nHttpd options:\n");
	printf("  -D, --httpd-dir=DIR       httpd dir\n");
	printf("  -P, --httpd-port=PORT     port number (default:8080)\n");
	printf("  -L, --httpd-listen=ADDR   listen address (default:localhost)\n");
	printf("\nInit options:\n");
	printf("  -i, --init=DIR            location to be initialized\n");
}

static void
syncVariable(List **vars, const char *name, char **value)
{
	if (*value == NULL)
		*value = pgut_strdup(getVariable(*vars, name));
	else
		*vars = addVariable(*vars, name, *value);
}

int
main(int argc, char *argv[])
{
	int	n = pgut_getopt(argc, argv, options);

	for (; n < argc; n++)
	{
		if (input == NULL)
			input = argv[n];
		else if (output == NULL)
			output = argv[n];
		else
			ereport(ERROR,
				(errcode(EINVAL),
				 errmsg("too many argumetns")));
	}

	if (init != NULL)
	{
		/* --init */
		return init_main(init, argv[0]);
	}
	else if (httpd_dir != NULL)
	{
		/* httpd mode */
		return httpd_main(httpd_dir, httpd_port, httpd_listen);
	}
	else if (input != NULL)
	{
		/* report mode */
		List *vars;

		vars = makeVariables(parameters);
		syncVariable(&vars, "dbname", &dbname);
		syncVariable(&vars, "host", &host);
		syncVariable(&vars, "port", &port);
		syncVariable(&vars, "username", &username);

		makeReport(NULL, input, vars);

		list_destroy(vars, freeVariable);
	}
	else
	{
		/* cgi mode */
		const char *document_root;

		if ((document_root = getenv("DOCUMENT_ROOT")) == NULL)
		{
			/* no command line and no environment variables -- maybe misusage */
			help(false);
			return 1;
		}

		return cgi_main(document_root);
	}

	return 0;
}

const char *
makeReport(StringInfo out, const char *xmlPath, List *vars)
{
	const char	   *result;
	xmlDocPtr		xml = NULL;

	if (xmlPath == NULL)
	{
		elog(WARNING, "no template");
		result = "*404";
		goto done;
	}

	if ((xml = xmlParseFile(xmlPath)) == NULL)
	{
		elog(WARNING, "failed to parse xml \"%s\"", xmlPath);
		result = "*500";
		goto done;
	}

	/* register functions */
	xsltRegisterExtModuleFunction(
		(xmlChar *) "query", (xmlChar *) PROGRAM_URL, pg_query);

	result = applyStylesheet(out, xml, stylesheet, vars);

done:
	xmlFreeDoc(xml);
	xsltCleanupGlobals();
	xmlCleanupParser();
	disconnect();

	return result;
}

static bool
removeNonDefaultNamespaces(xmlDocPtr xml)
{
	xmlNsPtr	ns;
	xmlNodePtr	html = xmlDocGetRootElement(xml);

	if (html == NULL)
		return false;

	for (ns = html->ns; ns; ns = ns->next)
	{
		/* default namespace has null prefix */
		if (ns->prefix == NULL)
		{
			html->ns = ns;
			ns->next = NULL;
			break;
		}
	}

	return true;
}

const char *
applyStylesheet(StringInfo out, xmlDocPtr xml, const char *xslPath, List *vars)
{
	xsltStylesheetPtr	xsl = NULL;
	xmlDocPtr			ret = NULL;
	const char		   *result = "*500";

	/* use default stylesheet if not specified */
	if (xslPath == NULL)
		xslPath = XSL_DEFAULT;

	if ((xsl = xsltParseStylesheetFile((xmlChar *) xslPath)) == NULL)
	{
		elog(WARNING, "failed to parse stylesheet \"%s\"", xslPath);
		goto done;
	}

	/* setup variables */
	if (pgut_echo)
	{
		ListCell *cell;
		foreach (cell, vars)
		{
			Variable *var = lfirst(cell);
			elog(LOG, "var: %s=%s", var->name, var->value);
		}
	}

	/* apply stylesheet to xml and write the result to file. */
	pg_query_vars = vars;
	if ((ret = xsltApplyStylesheet(xsl, xml, NULL)) == NULL)
	{
		elog(WARNING, "failed to apply stylesheet");
		goto done;
	}

	/*
	 * Cleanup namespace. Remove all namespaces except default.
	 * This might be a bad manner, but libxslt remains unused namespaces...
	 */
	if (!removeNonDefaultNamespaces(ret))
	{
		elog(WARNING, "failed to apply stylesheet");
		result = "*401";
		goto done;
	}
	if (out)
	{
		xmlChar	   *txt;
		int			len;
		if (xsltSaveResultToString(&txt, &len, ret, xsl) < 0)
		{
			elog(WARNING, "failed to write to string");
			goto done;
		}
		appendBinaryStringInfo(out, (char *)txt, len);
		xmlFree(txt);
	}
	else if (output)
	{
		if (xsltSaveResultToFilename(output, ret, xsl, 0) < 0)
		{
			elog(WARNING, "failed to write file \"%s\"", output);
			goto done;
		}
	}
	else
	{
		if (xsltSaveResultToFile(stdout, ret, xsl) < 0)
		{
			elog(WARNING, "failed to write result to stdout");
			goto done;
		}
	}

	result = "text/html";

done:
	/* cleanup objects and connections */
	pg_query_vars = NIL;
	xsltFreeStylesheet(xsl);
	xmlFreeDoc(ret);

	return result;
}

typedef struct QueryParam
{
	int				num;
	List		   *vars;
	char		   *names[9];
	const char	   *values[9];
} QueryParam;

static void
initQueryParam(QueryParam *param, List *vars)
{
	memset(param, 0, sizeof(QueryParam));
	param->vars = vars;
}

static void
termQueryParam(QueryParam *param)
{
	int		i;
	for (i = 0; i < param->num; i++)
		free(param->names[i]);
}

static const char *
assignQueryParam(QueryParam *param, char *query, int skip, const char *name, int len)
{
	int			i;
	const char *key;
	const char *value;
	ListCell   *cell;

	/* First, search query vars already used. */
	for (i = 0; i < param->num; i++)
	{
		key = param->names[i];
		if (strncmp(key, name, len) == 0 && key[len] == '\0')
		{
			query[0] = '$';
			query[1] = (char) ('0' + i + 1);
			memmove(query + 2, query + skip, strlen(query + skip) + 1);
			return param->values[i];
		}
	}

	/* only 9 parameters are supported for now */
	if (param->num >= 9)
	{
		elog(WARNING, "too many query parameters");
		return NULL;
	}

	/* Second, search global vars and add it query vars. */
	value = NULL;
	foreach(cell, param->vars)
	{
		Variable *var = lfirst(cell);

		key = var->name;
		if (strncmp(key, name, len) == 0 && key[len] == '\0')
		{
			value = var->value;
			break;
		}
	}

	param->names[param->num] = strdup_with_len(name, len);
	param->values[param->num] = value;	/* still NULL if key is not found. */
	param->num++;
	query[0] = '$';
	query[1] = (char) ('0' + param->num);
	memmove(query + 2, query + skip, strlen(query + skip) + 1);

	return value;
}

static void
rewriteQuery(char *query, QueryParam *param)
{
	char	   *var = query;

	/* TODO: placeholder cannot be used in string literals. */
	while ((var = strchr(var, VARTAG)) != NULL)
	{
		if (var[1] == '{')	/* expect ${name} */
		{
			const char *name = var + 2;
			int			len = 1;

			while (IsIdentBody(name[len])) { len++; }
			if (name[len] == '}')
				assignQueryParam(param, var, len + 3, name, len);
		}
		else if (IsIdentHead(var[1]))	/* expect $name */
		{
			const char *name = var + 1;
			int			len = 1;

			while (IsIdentBody(name[len])) { len++; }
			assignQueryParam(param, var, len + 1, name, len);
		}
		var = var + 1;
	}
}

/*
 * execute query with vars
 *
 * Replace ${name} or $name pattern to query parameters.
 * @param vars	array of ( nVars * { name, value } )
 */
static PGresult *
execute_with_vars(const char *query, List *vars)
{
	PGresult   *rs;
	char	   *rewritten;
	QueryParam	param;

	/* special case for dynamic query */
	if (query[0] == '$')
	{
		const char *name;
		int			len;

		if (query[1] == '{')	/* expect ${name} */
		{
			name = query + 2;
			len = 1;
			while (IsIdentBody(name[len])) { len++; }
			if (name[len] != '}')
				name = NULL;
		}
		else if (IsIdentHead(query[1]))	/* expect $name */
		{
			name = query + 1;
			len = 1;
			while (IsIdentBody(name[len])) { len++; }
		}
		else
		{
			name = NULL;
		}

		if (name)
		{
			ListCell *cell;
			foreach(cell, vars)
			{
				Variable *var = lfirst(cell);

				if (strncmp(var->name, name, len) == 0 && var->name[len] == '\0')
					return execute(var->value, 0, NULL);
			}
		}
	}

	rewritten = strdup_trim(query);
	initQueryParam(&param, vars);
	rewriteQuery(rewritten, &param);
	rs = execute(rewritten, param.num, param.values);
	free(rewritten);
	termQueryParam(&param);

	return rs;
}

/*
 * pg_query - returns query result as a table node set.
 *
 * FIXME: pg_query_vars should be a context variable but not a global variable.
 */
static void
pg_query(xmlXPathParserContextPtr ctxt, int nargs)
{
	xmlXPathObjectPtr	args[2];
	const char		   *values[2];
	PGresult		   *rs;
	int					i;

	/* first argument is query text. ramains are bind vars. */
	if (nargs <= 0)
		return;	/* no result */
	if (nargs > 2)
	{
		elog(WARNING, "too many args for pg:query() : %d passed", nargs);
		return;	/* no result */
	}
	for (i = nargs - 1; i >= 0; i--)
	{
		args[i] = valuePop(ctxt);
		if (args[i]->type != XPATH_STRING)
			args[i] = xmlXPathConvertString(args[i]);
		values[i] = (const char*)args[i]->stringval;
	}

	/* client encoding is always UTF8 because xmlChar is UTF8. */
	if (connection == NULL)
	{
		reconnect(WARNING);
		if (connection == NULL)
			goto done;
		PQsetClientEncoding(connection, "UTF8");
	}

	/* execute query in read-only transaction */
	command("BEGIN READ ONLY", 0, NULL);
	rs = execute_with_vars(values[0], pg_query_vars);
	if (PQresultStatus(rs) != PGRES_TUPLES_OK)
	{
		pgut_rollback(connection);
	}
	else
	{
		xmlNsPtr		   *ns;
		const Format	   *format;
		xmlXPathObjectPtr	ret;

		pgut_commit(connection);

		ns = xmlGetNsList(ctxt->context->doc, ctxt->context->node);
		while (ns && *ns)
		{
			if (strcmp((char*)(*ns)->href, NS_XHTML) == 0)
				break;
			ns++;
		}

		/* format the query result with format function */
		format = formats;
		if (nargs > 1)
		{
			for (i = 0; i < lengthof(formats); i++)
			{
				if (pg_strcasecmp(values[1], formats[i].name) == 0)
				{
					format = &formats[i];
					break;
				}
			}
			if (i >= lengthof(formats))
				elog(WARNING, "invalid format '%s'", values[1]);
		}
		ret = format->func(rs, (ns ? *ns : NULL), format->name);
		if (ret != NULL)
			valuePush(ctxt, ret);
	}

	PQclear(rs);

done:
	for (i = 0; i < nargs; i++)
		xmlXPathFreeObject(args[i]);
}

/*
 * <table>
 *   <thead>
 *     <tr>
 *       <td class="asc">ID</td>
 *       <td class="string">string-column</td>
 *       <td class="number">number-column</td>
 *     </tr>
 *   </thead>
 *   <tbody>
 *     <tr>
 *       <td class="number">ROWNUM</td>
 *       <td class="string">string-value</td>
 *       <td class="number">number-value</td>
 *     </tr>
 *     ...
 *   </tbody>
 * </table>
 */
static xmlXPathObjectPtr
createTABLE(PGresult *rs, xmlNsPtr ns, const char *name)
{
	xmlNodePtr			table, thead, tbody, tr;
	int					r, rows;
	int					c, cols;

	rows = PQntuples(rs);
	cols = PQnfields(rs);

	table = xmlNewNode(ns, (const xmlChar *) "table");

	/* thead */
	thead = xmlNewChild(table, NULL, (const xmlChar *) "thead", NULL);
	tr = xmlNewChild(thead, NULL, (const xmlChar *) "tr", NULL);
	addNode(tr, "th", "ID", ROWNUMOID);
	for (c = 0; c < cols; c++)
		addNode(tr, "th", PQfname(rs, c), PQftype(rs, c));

	/* tbody */
	tbody = xmlNewChild(table, NULL, (const xmlChar *) "tbody", NULL);
	for (r = 0; r < rows; r++)
	{
		char	rownum[32];

		/* <tr> */
		tr = xmlNewChild(tbody, NULL, (const xmlChar *) "tr", NULL);
		/* <td class="number">ROWNUM</td> */
		snprintf(rownum, lengthof(rownum), "%d", r + 1);
		addNode(tr, "td", rownum, INT4OID);
		/* <td class="number|string"> */
		for (c = 0; c < cols; c++)
			addNode(tr, "td", PQgetvalue(rs, r, c), PQftype(rs, c));
	}

	return xmlXPathNewNodeSet(table);
}

/*
 * <tbody>
 *   <tr>
 *     <th class="string">string-column</th>
 *     <td class="string">string-value</td>
 *     <td class="number">number-value</td>
 *   </tr>
 *   ...
 * </tbody>
 */
static xmlXPathObjectPtr
createTBODY(PGresult *rs, xmlNsPtr ns, const char *name)
{
	xmlNodePtr			tbody, tr;
	int					r, rows;
	int					c, cols;

	rows = PQntuples(rs);
	cols = PQnfields(rs);

	tbody = xmlNewNode(ns, (const xmlChar *) "tbody");
	for (c = 0; c < cols; c++)
	{
		/* <tr> */
		tr = xmlNewChild(tbody, NULL, (const xmlChar *) "tr", NULL);
		/* <th class="number|string">ROWNUM</th> */
		addNode(tr, "th", PQfname(rs, c), PQftype(rs, c));
		/* <td class="number|string"> */
		for (r = 0; r < rows; r++)
			addNode(tr, "td", PQgetvalue(rs, r, c), PQftype(rs, c));
	}

	return xmlXPathNewNodeSet(tbody);
}

/*
 * <dl>
 *   <dt>column</dt>
 *   <dd>value</dd>
 *   ...
 * </dl>
 */
static xmlXPathObjectPtr
createDL(PGresult *rs, xmlNsPtr ns, const char *name)
{
	xmlNodePtr			dl;
	int					c, cols;

	cols = PQnfields(rs);

	dl = xmlNewNode(ns, (const xmlChar *) "dl");
	for (c = 0; c < cols; c++)
	{
		addNode(dl, "dt", PQfname(rs, c), 0);
		addNode(dl, "dd", PQgetvalue(rs, 0, c), PQftype(rs, c));
	}

	return xmlXPathNewNodeSet(dl);
}

/*
 * <ul or ol>
 *   <li>value</li>
 *   ...
 * </ul or ol>
 */
static xmlXPathObjectPtr
createULOL(PGresult *rs, xmlNsPtr ns, const char *name)
{
	xmlNodePtr		ul;
	int				r, rows;

	rows = PQntuples(rs);

	ul = xmlNewNode(ns, (const xmlChar *) name);
	for (r = 0; r < rows; r++)
		addNode(ul, "li", PQgetvalue(rs, r, 0), PQftype(rs, 0));

	return xmlXPathNewNodeSet(ul);
}

/*
 * <span>value, ...</span>
 */
static xmlXPathObjectPtr
createSPAN(PGresult *rs, xmlNsPtr ns, const char *name)
{
	StringInfoData	str;
	xmlNodePtr		span;
	int				c, cols;

	initStringInfo(&str);
	cols = PQnfields(rs);

	span = xmlNewNode(ns, (const xmlChar *) name);
	for (c = 0; c < cols; c++)
	{
		if (c > 0)
			appendStringInfoString(&str, ", ");
		appendStringInfoString(&str, PQgetvalue(rs, 0, c));
	}
	xmlNodeAddContent(span, (const xmlChar *) str.data);

	termStringInfo(&str);
	return xmlXPathNewNodeSet(span);
}

static int
displen(const char *str)
{
	int		len;

	if (!*str)
		return 0;

	len = 0;
	for (;;)
	{
		int mlen = PQmblen(str, PG_UTF8);
		len += PQdsplen(str, PG_UTF8);
		for (; mlen > 0; mlen--)
		{
			++str;
			if (!*str)
				return len;
		}
	}
}

static void
appendLine(StringInfo str, int cols, const int width[], char sep, bool border)
{
	int		i, c;

	if (sep == '|')
		sep = '+';

	if (border)
		appendStringInfoChar(str, sep);
	for (c = 0; c < cols; c++)
	{
		if (c > 0)
			appendStringInfoChar(str, sep);
		for (i = 0; i < width[c] + 2; i++)
			appendStringInfoChar(str, '-');
	}
	if (border)
		appendStringInfoChar(str, sep);
	appendStringInfoChar(str, '\n');
}

/*
 * <pre>value, ...</pre>
 */
static xmlXPathObjectPtr
createPRE(PGresult *rs, xmlNsPtr ns, const char *name, char sep, bool border)
{
	xmlNodePtr		pre;
	int				r, rows;
	int				c, cols;
	int			   *width;
	StringInfoData	str;
	StringInfoData	fmt;

	initStringInfo(&str);
	initStringInfo(&fmt);
	rows = PQntuples(rs);
	cols = PQnfields(rs);
	width = malloc(cols * sizeof(int));

	/* calc width */
	for (c = 0; c < cols; c++)
	{
		width[c] = displen(PQfname(rs, c));

		for (r = 0; r < rows; r++)
		{
			int	w = displen(PQgetvalue(rs, r, c));
			if (width[c] < w)
				width[c] = w;
		}
	}

	/* header */
	if (border)
		appendLine(&str, cols, width, sep, border);
	if (border)
		appendStringInfoChar(&str, sep);
	for (c = 0; c < cols; c++)
	{
		const char *value = PQfname(rs, c);
		int			i = width[c] - displen(value);
		int			spcL = i / 2;
		int			spcR = (i + 1) / 2;

		if (c > 0)
			appendStringInfoChar(&str, sep);

		for (i = 0; i < spcL + 1; i++)
			appendStringInfoChar(&str, ' ');
		appendStringInfoString(&str, value);
		for (i = 0; i < spcR + 1; i++)
			appendStringInfoChar(&str, ' ');
	}
	if (border)
		appendStringInfoChar(&str, sep);
	appendStringInfoChar(&str, '\n');
	appendLine(&str, cols, width, sep, border);

	/* body */
	for (r = 0; r < rows; r++)
	{
		if (r > 0)
			appendStringInfoChar(&str, '\n');

		if (border)
			appendStringInfoChar(&str, sep);
		for (c = 0; c < cols; c++)
		{
			const char *value = PQgetvalue(rs, r, c);
			int			w = width[c] + strlen(value) - displen(value);

			if (c > 0)
				appendStringInfoChar(&str, sep);

			switch (PQftype(rs, c))
			{
				case INT2OID:
				case INT4OID:
				case INT8OID:
				case FLOAT4OID:
				case FLOAT8OID:
				case NUMERICOID:
				case OIDOID:
				case XIDOID:
				case CIDOID:
					/* right align */
					printfStringInfo(&fmt, " %%%ds ", w);
					break;
				default:
					/* left align */
					printfStringInfo(&fmt, " %%-%ds ", w);
					break;
			}

			appendStringInfo(&str, fmt.data, value);
		}
		if (border)
			appendStringInfoChar(&str, sep);
	}
	appendStringInfoChar(&str, '\n');

	/* footer */
	if (border)
		appendLine(&str, cols, width, sep, border);
	appendStringInfo(&str, "(%d %s)", rows, (rows == 1 ? "row" : "rows"));

	free(width);
	pre = xmlNewNode(ns, (const xmlChar *) "pre");
	xmlNodeAddContent(pre, (const xmlChar *) str.data);
	termStringInfo(&str);
	termStringInfo(&fmt);
	return xmlXPathNewNodeSet(pre);
}

/*
 * <pre>value, ...</pre>
 */
static xmlXPathObjectPtr
createPRE_postgres(PGresult *rs, xmlNsPtr ns, const char *name)
{
	return createPRE(rs, ns, name, '|', false);
}

static xmlXPathObjectPtr
createPRE_mysql(PGresult *rs, xmlNsPtr ns, const char *name)
{
	return createPRE(rs, ns, name, '|', true);
}

static xmlXPathObjectPtr
createPRE_oracle(PGresult *rs, xmlNsPtr ns, const char *name)
{
	return createPRE(rs, ns, name, ' ', false);
}

static xmlXPathObjectPtr
createCIRCLE(PGresult *rs, xmlNsPtr ns, const char *name)
{
	xmlNodePtr		graph;
	StringInfoData	values;

	initStringInfo(&values);
	toArray(&values, rs);

	graph = xmlNewNode(NULL, (const xmlChar *) name);
	xmlNewTextChild(graph, NULL, (const xmlChar *) "values", (const xmlChar *) values.data);

	termStringInfo(&values);
	return xmlXPathNewNodeSet(graph);
}

static xmlXPathObjectPtr
createVBAR(PGresult *rs, xmlNsPtr ns, const char *name)
{
	xmlNodePtr		graph;
	StringInfoData	values;
	StringInfoData	vars;
	int				c, cols;

	cols = PQnfields(rs);

	initStringInfo(&values);
	toArray(&values, rs);

	initStringInfo(&vars);
	appendStringInfoString(&vars, "x: [null");
	for (c = 1; c < cols; c++)
		appendStringInfo(&vars, ", '%s'", PQfname(rs, c));
	appendStringInfoChar(&vars, ']');

	graph = xmlNewNode(NULL, (const xmlChar *) name);
	xmlNewTextChild(graph, NULL, (const xmlChar *) "values", (const xmlChar *) values.data);
	xmlNewTextChild(graph, NULL, (const xmlChar *) "vars", (const xmlChar *) vars.data);

	termStringInfo(&values);
	termStringInfo(&vars);
	return xmlXPathNewNodeSet(graph);
}

static double
ceilby(double v, double div)
{
	double	n = 1;
	v /= div;

	if (v <= DBL_MIN)
		return v;
	else if (v > 1)
		while (v > n * 10) { n *= 10; }
	else
		while (v < n * 10) { n /= 10; }

	return ceil(v / n) * n;
}

#define LINE_NUM_Y		5

static void
appendVars(StringInfo vars, double vMin, double vMax)
{
	double		yMin, yMax, dy;

	/*
	 * When the difference is too small, use yMin and yMax from value range.
	 * Otherwise, expand display range to include zero.
	 */
	if (vMax - vMin < Max(Abs(vMin), Abs(vMax)) / 10)
	{
		dy = ceilby(vMax - vMin, LINE_NUM_Y);
		yMax = (floor(vMax / dy) + 1) * dy;
		yMin = (ceil(vMin / dy) - 1) * dy;
	}
	else
	{
		yMax = Max(0, vMax);
		yMin = Min(0, vMin);
		dy = ceilby(yMax - yMin, LINE_NUM_Y);
		if (vMax > 0)
			yMax = (floor(vMax / dy) + 1) * dy;
		if (vMin < 0)
			yMin = (ceil(vMin / dy) - 1) * dy;
	}

	appendStringInfoString(vars, "dLabel: false");
	/* yMax & yMin */
	appendStringInfo(vars, ",\nyMax: %f", yMax);
	appendStringInfo(vars, ",\nyMin: %f", yMin);
	/* y */
	if (vMin != vMax)
	{
		int		i;

		appendStringInfoString(vars, ",\ny: [''");
		for (i = (int) floor(yMin / dy); dy * i <= yMax; i++)
			appendStringInfo(vars, ", %f", i * dy);
		appendStringInfoChar(vars, ']');
	}
}

static void
createLINE1D(PGresult *rs, StringInfo values, StringInfo vars)
{
	int				r, rows;
	int				c, cols;
	double			vMin = +DBL_MAX;
	double			vMax = -DBL_MAX;

	rows = PQntuples(rs);
	cols = PQnfields(rs);

	/* start column with 1. The first column is name of x-axis. */
	for (c = 1; c < cols; c++)
	{
		if (c > 1)
			appendStringInfoString(values, ",\n");
		appendStringInfo(values, "['%s'", PQfname(rs, c));
		for (r = 0; r < rows; r++)
		{
			const char *v = PQgetvalue(rs, r, c);
			double		y = atof(v);
			vMin = Min(vMin, y);
			vMax = Max(vMax, y);
			appendStringInfo(values, ", %s", v);
		}
		appendStringInfoChar(values, ']');
	}

	appendVars(vars, vMin, vMax);
	/* x */
	appendStringInfo(vars, ",\nx: ['%s'", PQfname(rs, 0));
	for (r = 0; r < rows; r++)
		appendStringInfo(vars, ", '%s'", PQgetvalue(rs, r, 0));
	appendStringInfoChar(vars, ']');
}

typedef struct Line
{
	char		   *name;
	int				len;
	StringInfoData *buf;
} Line;

#define MAX_LINES		10

static void
createLINE2D(PGresult *rs, StringInfo values, StringInfo vars)
{
	StringInfoData	xValues;
	char			prev[1024] = "";
	Line			lines[MAX_LINES];
	int				n, nlines;
	int				r, rows;
	int				c, cols;
	int				len;
	double			vMin = +DBL_MAX;
	double			vMax = -DBL_MAX;

	rows = PQntuples(rs);
	cols = PQnfields(rs);

	initStringInfo(&xValues);
	memset(lines, 0, sizeof(Line) * MAX_LINES);

	len = 0;
	nlines = 0;
	for (r = 0; r < rows; r++)
	{
		char		x[1024];
		char		name[1024];

		if (sscanf(PQgetvalue(rs, r, 0), "(%1024[^,],%1024[^)])", x, name) != 2)
		{
			elog(WARNING, "parse error [%d] : %s", r, PQgetvalue(rs, r, 0));
			continue;
		}

		if (strcmp(prev, x) != 0)
		{
			len++;
			strlcpy(prev, x, lengthof(prev));
			appendStringInfo(&xValues, ", '%s'", x);
		}

		for (n = 0; n < nlines && strcmp(name, lines[n].name) != 0; n++)
			;
		if (n >= MAX_LINES)
		{
			elog(WARNING, "ignore lines: %s", name);
			continue;
		}
		if (lines[n].name == NULL)
		{
			/* not found */
			nlines = n + 1;
			lines[n].name = pgut_strdup(name);
			lines[n].buf = pgut_newarray(StringInfoData, cols - 1);
			for (c = 0; c < cols - 1; c++)
				initStringInfo(&lines[n].buf[c]);
			while (lines[n].len < len - 1)
			{
				for (c = 0; c < cols - 1; c++)
					appendStringInfoString(&lines[n].buf[c], ", 0");
				lines[n].len++;
			}
		}

		for (c = 0; c < cols - 1; c++)
		{
			const char *v = PQgetvalue(rs, r, c + 1);
			double		y = atof(v);
			int			i;

			vMin = Min(vMin, y);
			vMax = Max(vMax, y);

			for (i = lines[n].len; i < len; i++)
				appendStringInfo(&lines[n].buf[c], ", %s", v);
		}
		lines[n].len = len;
	}

	/* concat lines into values */
	initStringInfo(values);
	for (n = 0; n < nlines; n++)
	{
		for (c = 0; c < cols - 1; c++)
		{
			if (n > 0 || c > 0)
				appendStringInfoString(values, ",\n");
			if (cols > 2)
				appendStringInfo(values, "['%s %s'%s]",
					lines[n].name, PQfname(rs, c + 1), lines[n].buf[c].data);
			else
				appendStringInfo(values, "['%s'%s]",
					lines[n].name, lines[n].buf[c].data);
		}
	}

	appendVars(vars, vMin, vMax);
	/* x */
	appendStringInfo(vars, ",\nx: ['%s'%s]", PQfname(rs, 0), xValues.data);

	termStringInfo(&xValues);
	for (n = 0; n < nlines; n++)
	{
		free(lines[n].name);
		for (c = 0; c < cols - 1; c++)
			termStringInfo(&lines[n].buf[c]);
	}
}

/* true if the first column is row type */
static bool
is2D(PGresult *rs)
{
	return PQnfields(rs) > 1 &&
		   PQntuples(rs) > 0 &&
		   PQftype(rs, 0) != TEXTOID &&
		   PQgetvalue(rs, 0, 0)[0] == '(';
}

static xmlXPathObjectPtr
createLINE(PGresult *rs, xmlNsPtr ns, const char *name)
{
	xmlNodePtr		graph;
	StringInfoData	values;
	StringInfoData	vars;

	initStringInfo(&values);
	initStringInfo(&vars);
	if (is2D(rs))
		createLINE2D(rs, &values, &vars);
	else
		createLINE1D(rs, &values, &vars);

	graph = xmlNewNode(NULL, (const xmlChar *) name);
	xmlNewTextChild(graph, NULL, (const xmlChar *) "values", (const xmlChar *) values.data);
	xmlNewTextChild(graph, NULL, (const xmlChar *) "vars", (const xmlChar *) vars.data);

	termStringInfo(&values);
	termStringInfo(&vars);
	return xmlXPathNewNodeSet(graph);
}

static xmlXPathObjectPtr
createRADAR(PGresult *rs, xmlNsPtr ns, const char *name)
{
	xmlNodePtr		graph;
	StringInfoData	values;
	StringInfoData	vars;
	int				c, cols;

	cols = PQnfields(rs);

	initStringInfo(&values);
	toArray(&values, rs);

	initStringInfo(&vars);
	appendStringInfoString(&vars, "aCap: [");
	for (c = 1; c < cols; c++)
	{
		if (c > 1)
			appendStringInfoString(&vars, ", ");
		appendStringInfo(&vars, "'%s'", PQfname(rs, c));
	}
	appendStringInfoChar(&vars, ']');

	graph = xmlNewNode(NULL, (const xmlChar *) name);
	xmlNewTextChild(graph, NULL, (const xmlChar *) "values", (const xmlChar *) values.data);
	xmlNewTextChild(graph, NULL, (const xmlChar *) "vars", (const xmlChar *) vars.data);

	termStringInfo(&values);
	termStringInfo(&vars);
	return xmlXPathNewNodeSet(graph);
}

static void
toArray(StringInfo str, PGresult *rs)
{
	int					r, rows;
	int					c, cols;

	rows = PQntuples(rs);
	cols = PQnfields(rs);

	for (r = 0; r < rows; r++)
	{
		if (r > 0)
			appendStringInfoString(str, ",\n");
		appendStringInfo(str, "['%s'", PQgetvalue(rs, r, 0));
		for (c = 1; c < cols; c++)
			appendStringInfo(str, ", %s", PQgetvalue(rs, r, c));
		appendStringInfoChar(str, ']');
	}
}

static void
addNode(xmlNodePtr parent, const char *name, const char *value, Oid type)
{
	xmlNodePtr	cell;
	const char *typname;

	switch (type)
	{
		case 0:			/* no typname */
			typname = NULL;
			break;
		case ROWNUMOID:	/* special case for default sort */
			typname = "asc";
			break;
		case INT2OID:
		case INT4OID:
		case INT8OID:
		case FLOAT4OID:
		case FLOAT8OID:
		case NUMERICOID:
		case OIDOID:
		case XIDOID:
		case CIDOID:
			typname = "number";
			break;
#ifdef NOT_USED
		case DATEOID:
		case TIMEOID:
		case TIMESTAMPOID:
		case TIMESTAMPTZOID:
			typname = "date";
			break;
#endif
		default:
			typname = "string";
			break;
	}

	cell = xmlNewChild(parent, NULL, (const xmlChar *) name, NULL);
	if (typname)
		xmlNewProp(cell, (const xmlChar *) "class", (const xmlChar *) typname);

	if (type == XMLOID)
	{
		while (IsSpace(*value)) { value++; }
		if (*value == '<')
		{
			xmlDocPtr	child = xmlParseMemory(value, strlen(value));
			if (child != NULL)
			{
				xmlNodePtr	root = xmlDocGetRootElement(child);
				xmlUnlinkNode(root);
				xmlAddChild(cell, root);
				xmlFreeDoc(child);
				return;
			}
			else
				elog(WARNING, "failed to parse: %s", value);
		}
	}

	xmlNodeAddContent(cell, (const xmlChar *) value);
}
