/*-------------------------------------------------------------------------
 *
 * interpreter.c		- Interpterer for the PL/pgPSM
 *			  procedural language
 *
 * Portions Copyright (c) 1996-2012, PostgreSQL Global Development Group
 * Portions Copyright (c) 1994, Regents of the University of California
 *
 *
 * IDENTIFICATION
 *	  src/pl/plpgpsm/interpreter.c
 *
 *-------------------------------------------------------------------------
 */

#include "plpgpsm.h"

#include "funcapi.h"

#include "access/xact.h"
#include "catalog/pg_proc.h"
#include "catalog/pg_type.h"
#include "utils/builtins.h"
#include "utils/datum.h"

static PLpgPSM_result_code_enum eval_astnode(PLpgPSM_execstate *estate, PLpgPSM_astnode *astnode);
static int eval_handler(PLpgPSM_execstate *estate, PLpgPSM_declare_handler *handler);
static void push_edata(PLpgPSM_execstate *estate, ErrorData *edata);
static ErrorData *pop_edata(PLpgPSM_execstate *estate);


/* ----------
 * plpgpsm_exec_function	Called by the call handler for
 *				function execution.
 * ----------
 */
Datum
plpgpsm_exec_function(PLpgPSM_function *func, FunctionCallInfo fcinfo)
{
	PLpgPSM_result_code_enum rc;
	Datum	result = (Datum) 0;
	Oid		rettypeid;
	PLpgPSM_execstate  estate;

	plpgpsm_init_execstate(&estate, func, fcinfo);

	rettypeid = get_fn_expr_rettype(fcinfo->flinfo);
	if (!OidIsValid(rettypeid))
		elog(ERROR, "cannot to determine actual return type");

	estate.rtinfo = plpgpsm_make_typeinfo_typeid(&estate, rettypeid, -1);
	if (func->parser->noutargs > 1)
		get_call_result_type(fcinfo, NULL, &estate.rettupdesc);
	else
		estate.rettupdesc = NULL;

	rc = eval_astnode(&estate, func->astroot);

	if (rc != PLPGPSM_RC_RETURN && rettypeid != VOIDOID && func->parser->noutargs == 0)
	{
		ereport(ERROR,
			   (errcode(ERRCODE_S_R_E_FUNCTION_EXECUTED_NO_RETURN_STATEMENT),
				errmsg("control reached end of function without RETURN")));
	}

	/*
	 * If the function's return type isn't by value, copy the value
	 * into upper executor memory context.
	 *
	 */
	if (func->parser->noutargs == 1)
	{
		ListCell	*lc;

		foreach(lc, ((PLpgPSM_eb_stmt *) func->astroot)->stmts)
		{
			PLpgPSM_declare_param *declpar = (PLpgPSM_declare_param *) lfirst(lc);

			if (declpar->astnode.type != PLPGPSM_DECLARE_PARAM)
				break;

			if (declpar->param_mode == PROARGMODE_OUT || declpar->param_mode == PROARGMODE_INOUT)
			{
				PLpgPSM_variable *var = declpar->variable;
				PLpgPSM_datum *datum = &estate.datums[var->offset];

				estate.retisnull = datum->isnull;
				estate.retval  = datum->value;

				break;
			}
		}
	}
	else if (func->parser->noutargs > 1)
	{
		Datum *values = (Datum *) palloc(func->parser->noutargs * sizeof(Datum));
		bool *nulls = (bool *) palloc(func->parser->noutargs * sizeof(bool));
		ListCell	*lc;
		int	i = 0;
		HeapTuple	tuple;

		foreach(lc, ((PLpgPSM_eb_stmt *) func->astroot)->stmts)
		{
			PLpgPSM_declare_param *declpar = (PLpgPSM_declare_param *) lfirst(lc);

			if (declpar->astnode.type != PLPGPSM_DECLARE_PARAM)
				break;

			if (declpar->param_mode == PROARGMODE_OUT || declpar->param_mode == PROARGMODE_INOUT)
			{
				PLpgPSM_variable *var = declpar->variable;
				PLpgPSM_datum *datum = &estate.datums[var->offset];

				values[i] = datum->value;
				nulls[i++] = datum->isnull;
			}
		}

		tuple = heap_form_tuple(estate.rettupdesc, values, nulls);
		estate.retisnull = false;
		estate.retval = HeapTupleGetDatum(tuple);
	}

	result = estate.retval;
	fcinfo->isnull = estate.retisnull;

	if (!estate.retisnull && !estate.rtinfo->typbyval)
	{
		Size		len;
		void	   *tmp;

		len = datumGetSize(result, false, estate.rtinfo->typlen);
		tmp = SPI_palloc(len);
		memcpy(tmp, DatumGetPointer(result), len);
		result = PointerGetDatum(tmp);
	}

	plpgpsm_clean_execstate(&estate);

	/*
	 * Return the function's result
	 */
	return result;
}

static PLpgPSM_result_code_enum
eval_stmts(PLpgPSM_execstate *estate, PLpgPSM_astnode *astnode, bool *leave)
{
	PLpgPSM_eb_stmt *stmt = (PLpgPSM_eb_stmt *) astnode;
	ListCell	*lc;
	PLpgPSM_result_code_enum rc;

	Assert (plpgpsm_allow_cast_to_eb(astnode));

	*leave = false;

	foreach (lc, stmt->stmts)
	{
		rc = eval_astnode(estate, (PLpgPSM_astnode *) lfirst(lc));
		switch (rc)
		{
			case PLPGPSM_RC_OK:
				/* all is ok - do nothing */
				break;

			case PLPGPSM_RC_RETURN:
				*leave = true;
				return PLPGPSM_RC_RETURN;

			case PLPGPSM_RC_LEAVE:
				*leave = true;
				if (estate->stop_node == astnode)
				{
					estate->stop_node = NULL;

					return PLPGPSM_RC_OK;
				}
				else
					return PLPGPSM_RC_LEAVE;

			case PLPGPSM_RC_ITERATE:
				if (estate->stop_node == astnode)
				{
					Assert(astnode->type != PLPGPSM_STMT_COMPOUND);

					estate->stop_node = NULL;

					return PLPGPSM_RC_OK;
				}
				else
				{
					*leave = true;
					return PLPGPSM_RC_ITERATE;
				}

			case PLPGPSM_RC_UNDO:
				if (estate->stop_node == astnode)
				{
					Assert(astnode->type == PLPGPSM_STMT_COMPOUND);
					estate->stop_node = NULL;

					return PLPGPSM_RC_UNDO;
				}
				else
				{
					*leave = true;

					return PLPGPSM_RC_UNDO;
				}
		}
	}

	return PLPGPSM_RC_OK;
}

/*
 * sqlcode should be integer, sqlstate char(5), or varchar(5) or text
 *
 */
static void
special_variables_check_type(PLpgPSM_variable *var, PLpgPSM_typeinfo *typeinfo)
{
	if (typeinfo != NULL)
	{
		if (var->special == PLPGPSM_SPECIAL_SQLCODE && typeinfo->typeid != INT4OID)
			elog(ERROR, "SQLCODE should be declared as integer");

		if (var->special == PLPGPSM_SPECIAL_SQLSTATE)
		{
			/* length must 9 or -1 */
			if (typeinfo->typmod != -1 && typeinfo->typmod < 9)
				elog(ERROR, "SQLSTATE has minimal length 5 chars");
			if (typeinfo->typeid != BPCHAROID && typeinfo->typeid != VARCHAROID && typeinfo->typeid != TEXTOID)
				elog(ERROR, "SQLSTATE should be char(n), varchar(n) or text type");
		}
	}
}


/*
 * closes cursor
 *
 *   when closed_ok is true, then it doesn't raise error, when cursor is closed yet.
 *
 */
static void
close_cursor(PLpgPSM_execstate *estate, PLpgPSM_cursor *cursor, bool closed_ok)
{
	PLpgPSM_datum *datum;

	Assert(cursor != NULL && cursor->astnode.type == PLPGPSM_CURSOR);
	datum = &estate->datums[cursor->offset];
	Assert(datum->class == PLPGPSM_DATUM_CURSOR);

	if (!datum->isnull)
	{
		Portal	portal = (Portal) DatumGetPointer(datum->value);

		UnpinPortal(portal);

		plpgpsm_close_cursor(estate, portal);
		datum->isnull = true;
		datum->value = (Datum) 0;

		if (datum->extra != NULL)
		{
			PLpgPSM_fetch_extra *extra = (PLpgPSM_fetch_extra *) datum->extra;

			if (extra->tuptab)
			{
				SPI_freetuptable(extra->tuptab);
				pfree(datum->extra);
				datum->extra = NULL;
			}
		}
	}
	else if (!closed_ok)
		elog(ERROR, "cursor \"%s\" is not opened", cursor->name);
}

/*
 * open cursor
 *
 */
static void
open_cursor(PLpgPSM_execstate *estate, PLpgPSM_cursor *cursor)
{
	PLpgPSM_datum *datum;
	Portal			portal;
	PLpgPSM_ESQL	*esql = NULL;
	int		options = 0;
	PLpgPSM_fetch_extra *extra = NULL;
	char		*name = NULL;

	Assert(cursor != NULL && cursor->astnode.type == PLPGPSM_CURSOR);

	datum = &estate->datums[cursor->offset];
	Assert(datum->class == PLPGPSM_DATUM_CURSOR);

	if (!datum->isnull)
		elog(NOTICE, "cursor \"%s\" is opened yet", cursor->name);

	if (cursor->astnode.parent->type == PLPGPSM_DECLARE_CURSOR)
	{
		esql = ((PLpgPSM_declare_cursor *) cursor->astnode.parent)->esql;
		options = ((PLpgPSM_declare_cursor *) cursor->astnode.parent)->option;
	}
	else if (cursor->astnode.parent->type == PLPGPSM_STMT_FOR)
	{
		esql = ((PLpgPSM_eb_stmt *) cursor->astnode.parent)->expr;
		options = 0;

		/* use name when is explicitly enterred */
		if (cursor->force_name)
			name = cursor->name;
	}
	else
		elog(ERROR, "unsupported parent of PLPGPSM_CURSOR astnode");

	portal = plpgpsm_open_cursor(estate, esql, name, options);
	PinPortal(portal);

	datum->value = PointerGetDatum(portal);
	datum->isnull = false;

	if (options == 0)
	{
		extra = (PLpgPSM_fetch_extra *) palloc(sizeof(PLpgPSM_fetch_extra));

		extra->tuptab = NULL;
		extra->prefetch = options == 0;
		extra->nattrs = -1;
		extra->values = NULL;
		extra->nulls = NULL;
		extra->cursor_tuple = NULL;
		extra->cursor_tupdesc = NULL;
		extra->is_ready = false;
	}

	datum->extra = extra;
}

/*
 * fetch from cursor
 *
 */
static void
fetch_from_cursor(PLpgPSM_execstate *estate, PLpgPSM_cursor *cursor, PLpgPSM_datum *datum)
{
	PLpgPSM_fetch_extra *extra;

	Assert(datum->class == PLPGPSM_DATUM_CURSOR);

	if (datum->isnull)
		elog(NOTICE, "cursor \"%s\" is not opened", cursor->name);

	extra = (PLpgPSM_fetch_extra *) datum->extra;

	if (extra && extra->tuptab != NULL)
	{
		if (extra->n < extra->processed && extra->processed > 0)
		{
			extra->cursor_tuple = extra->tuptab->vals[extra->n++];
			extra->cursor_tupdesc = extra->tuptab->tupdesc;
			extra->is_ready = true;
			estate->sqlcode = 0;

			return;
		}
		else
		{
			/* release prefetch */
			SPI_freetuptable(extra->tuptab);
			extra->tuptab = NULL;
			extra->is_ready = false;
		}
	}

	if (extra && extra->prefetch)
	{
		/* fill prefetch cache */
		plpgpsm_cursor_fetch(estate, (Portal) DatumGetPointer(datum->value), 50);

		extra->tuptab = estate->eval_tuptable;
		extra->processed = estate->eval_processed;
		extra->cursor_tupdesc = extra->tuptab->tupdesc;

		if (estate->eval_processed > 0)
		{
			extra->cursor_tuple = extra->tuptab->vals[0];
			estate->sqlcode = 0;
		}
		else
		{
			extra->cursor_tuple = NULL;
		}

		extra->n = 1;
		extra->is_ready = true;

		/* ensure don't double free SPI_tuptable */
		estate->eval_tuptable = NULL;
		estate->eval_processed = 0;
	}
	else
		plpgpsm_cursor_fetch(estate, (Portal) DatumGetPointer(datum->value), 1);

	return;
}

static void
declare_cursor_datum(PLpgPSM_cursor *cursor, PLpgPSM_datum *datum)
{
	/*
	 * there should be unclosed cursors for previous broken iterations.
	 * Related cursors will be closed, but there can be allocated prefetch
	 * cache, and this cache should be released first.
	 */
	if (!datum->isnull)
	{
		PLpgPSM_fetch_extra *extra = (PLpgPSM_fetch_extra *) datum->extra;

		if (extra != NULL)
		{
			if (extra->tuptab != NULL)
			{
				/* release prefetch */
				SPI_freetuptable(extra->tuptab);
				extra->tuptab = NULL;
			}
			pfree(extra);
		}
	}

	datum->class = PLPGPSM_DATUM_CURSOR;
	datum->typeinfo = NULL;
	datum->variable = (PLpgPSM_astnode *) cursor;
	datum->isnull = true;
	datum->value = (Datum) 0;
	datum->extra = NULL;
}

static void
close_cursors(PLpgPSM_execstate *estate, PLpgPSM_astnode *astnode)
{
	PLpgPSM_eb_stmt *stmt = (PLpgPSM_eb_stmt *) astnode;
	ListCell	*l;

	Assert(stmt->astnode.type == PLPGPSM_STMT_COMPOUND);
	foreach(l, stmt->stmts)
	{
		PLpgPSM_astnode *anode = (PLpgPSM_astnode *) lfirst(l);

		if (anode->type == PLPGPSM_DECLARE_VARIABLE)
		{
			continue;
		}
		else if (anode->type == PLPGPSM_DECLARE_CURSOR)
		{
			close_cursor(estate, ((PLpgPSM_declare_cursor *) anode)->cursor, true);
		}
		else
			break;
	}
}

static void
set_edata(ErrorData *edata, int elevel, int sqlstate,
						    char *message,
						    char *detail,
						    char *hint,
						    char *condition_name)
{
	memset(edata, 0, sizeof(ErrorData));

	edata->elevel = elevel;
	edata->sqlerrcode = sqlstate;
	edata->message = message;
	edata->detail = detail;
	edata->hint = hint;

	/*
	 * There are not field for condition name in ErrorData. For this moment
	 * we can use a detail_log field, but it is ugly and unsafe - should be
	 * part of PostgreSQL ToDo.
	 */
	edata->detail_log = condition_name;
}

/*
 * Does same work like FreeErrorData without free object
 *
 */
static void
free_edata(ErrorData *edata)
{
	if (edata->message)
		pfree(edata->message);
	if (edata->detail)
		pfree(edata->detail);
	if (edata->hint)
		pfree(edata->hint);
	if (edata->detail_log)
		pfree(edata->detail_log);
	if (edata->context)
		pfree(edata->context);
	if (edata->internalquery)
		pfree(edata->internalquery);
}

/*
 * evaluate ast node
 *
 */
static PLpgPSM_result_code_enum
eval_astnode(PLpgPSM_execstate *estate, PLpgPSM_astnode *astnode)
{
	PLpgPSM_result_code_enum	rc;

	estate->curr_stmt = astnode;

	switch (astnode->type)
	{
		case PLPGPSM_DECLARE_PARAM:
			{
				PLpgPSM_declare_param *decl_par = (PLpgPSM_declare_param *) astnode;
				PLpgPSM_variable *var = decl_par->variable;
				PLpgPSM_datum		*datum = &estate->datums[var->offset];

				datum->typeinfo = plpgpsm_make_typeinfo_typeid(estate, decl_par->typeid, -1);
				if (decl_par->param_mode == PROARGMODE_IN || decl_par->param_mode == PROARGMODE_INOUT)
					plpgpsm_set_datum(estate, datum,
								    estate->fcinfo->arg[decl_par->param_offset],
								    estate->fcinfo->argnull[decl_par->param_offset],
														    true);
				else
				{
					datum->isnull = true;
					datum->value = (Datum) 0;
				}

				/* recheck type for special variables */
				if (var->special != PLPGPSM_SPECIAL_NONE)
					special_variables_check_type(var, datum->typeinfo);

				return PLPGPSM_RC_OK;
			}

		case PLPGPSM_DECLARE_VARIABLE:
			{
				PLpgPSM_declare_variable *declvar = (PLpgPSM_declare_variable *) astnode;
				ListCell	*lc;
				PLpgPSM_typeinfo	*tinfo = NULL;
				bool		not_executed_yet = true;

				foreach (lc, declvar->variables)
				{
					PLpgPSM_variable *var = (PLpgPSM_variable *) lfirst(lc);
					PLpgPSM_datum	*datum = &estate->datums[var->offset];

					if (datum->typeinfo == NULL)
					{
						if (tinfo != NULL)
							datum->typeinfo = tinfo;
						else if (declvar->typename != NULL)
						{
							tinfo =  plpgpsm_make_typeinfo(estate, declvar->typename);
							datum->typeinfo = tinfo;
						}
					}

					if (declvar->expr != NULL)
					{
						if (not_executed_yet)
						{
							plpgpsm_eval_expr(estate, declvar->expr, tinfo, true);
							not_executed_yet = false;
						}
						plpgpsm_store_result(estate, 1, datum);

						/* special variables should not have a default value */
						if (var->special == PLPGPSM_SPECIAL_SQLCODE)
							elog(ERROR, "SQLCODE should not have a default value");
						else if (var->special == PLPGPSM_SPECIAL_SQLSTATE)
							elog(ERROR, "SQLSTATE should not have a default value");
					}
					else
						plpgpsm_set_datum(estate, datum, (Datum) 0, true, false);

					if (var->special != PLPGPSM_SPECIAL_NONE)
					{
						special_variables_check_type(var, datum->typeinfo);

						if (var->special == PLPGPSM_SPECIAL_SQLCODE)
							datum->class = PLPGPSM_DATUM_SQLCODE;
						else if (var->special == PLPGPSM_SPECIAL_SQLSTATE)
							datum->class = PLPGPSM_DATUM_SQLSTATE;
					}
					else
						datum->class = PLPGPSM_DATUM_SCALAR;
				}

				if (!not_executed_yet)
					plpgpsm_eval_cleanup(estate);

				return PLPGPSM_RC_OK;
			}

		case PLPGPSM_STMT_ASSIGN:
			{
				PLpgPSM_stmt_assign *stmt = (PLpgPSM_stmt_assign *) astnode;

				Assert(stmt->expr != NULL);

				if (stmt->target != NULL)
				{
					if (stmt->target->ref == NULL)
						elog(ERROR, "unjoined referer");

					if (stmt->target->ref->type == PLPGPSM_VARIABLE)
					{
						PLpgPSM_variable *var;
						PLpgPSM_datum *datum;

						var = (PLpgPSM_variable *) stmt->target->ref;
						datum = &estate->datums[var->offset];

						plpgpsm_eval_expr(estate, stmt->expr, datum->typeinfo, true);
						plpgpsm_store_result(estate, 1, datum);
						plpgpsm_eval_cleanup(estate);

						return PLPGPSM_RC_OK;
					}
				}
				else
				{
					Assert(stmt->target_list != NULL);

					plpgpsm_eval_expr(estate, stmt->expr, NULL, true);
					plpgpsm_store_composed_result_to_datums(estate, stmt->target_list);

					return PLPGPSM_RC_OK;
				}
			}

		case PLPGPSM_SIMPLE_BLOCK:
		case PLPGPSM_CONDITIONAL_BLOCK:
			{
				PLpgPSM_eb_stmt *stmt = (PLpgPSM_eb_stmt *) astnode;
				ListCell	*lc;

				Assert(stmt->label == NULL);

				/* simple block should not have label and expression */
				Assert(astnode->type != PLPGPSM_SIMPLE_BLOCK || stmt->expr == NULL);

				/* don't execute, when expr is not empty and is not true */
				if (stmt->expr != NULL && !plpgpsm_eval_boolean(estate, stmt->expr))
					return PLPGPSM_RC_OK;

				/* fast and simple way */
				foreach (lc, stmt->stmts)
				{
					if ((rc = eval_astnode(estate, (PLpgPSM_astnode *) lfirst(lc))) != PLPGPSM_RC_OK)
						return rc;
				}

				/*
				 * After successfull execution of conditional block we would to leave
				 * outer statement (should be IF or CASE)
				 */
				if (astnode->type == PLPGPSM_CONDITIONAL_BLOCK)
				{
					Assert (astnode->parent->type == PLPGPSM_STMT_IF ||
						astnode->parent->type == PLPGPSM_STMT_CASE);

					estate->stop_node = astnode->parent;
					return PLPGPSM_RC_LEAVE;
				}

				return PLPGPSM_RC_OK;
			}

		case PLPGPSM_STMT_IF:
		case PLPGPSM_STMT_CASE:
			{
				bool		leave;

				Assert(((PLpgPSM_eb_stmt *) astnode)->expr == NULL && 
				       ((PLpgPSM_eb_stmt *) astnode)->label == NULL);
				if ((rc = eval_stmts(estate, astnode, &leave)) != PLPGPSM_RC_OK)
					return rc;

				/*
				 * Only sucessfully executed condition block can to set
				 * leave flag. Raise error, when any condition bloc was not
				 * executed.
				 *
				 */
				if (astnode->type == PLPGPSM_STMT_CASE && !leave)
				{
					estate->sqlcode = MAKE_SQLSTATE('0','2','0','0','0');

					if (((PLpgPSM_eb_stmt *) astnode)->handler != NULL)
						return eval_handler(estate,
									((PLpgPSM_eb_stmt *) astnode)->handler);
				}

				return PLPGPSM_RC_OK;
			}

		case PLPGPSM_STMT_COMPOUND:
			{
				PLpgPSM_eb_stmt *stmt = (PLpgPSM_eb_stmt *) astnode;
				bool		leave;

				if (!stmt->option)
				{
					/* non atomic compound statement */
					rc = eval_stmts(estate, astnode, &leave);

					/* close opened cursors */
					close_cursors(estate, astnode);

					if (leave)
						return rc;
				}
				else
				{
					ResourceOwner		oldowner;
					MemoryContext		oldCxt;

					/* atomic compound statement */

					oldCxt = CurrentMemoryContext;
					oldowner = CurrentResourceOwner;

					PG_TRY();
					{
						BeginInternalSubTransaction(NULL);
						MemoryContextSwitchTo(oldCxt);

						rc = eval_stmts(estate, astnode, &leave);

						/* close opened cursors */
						close_cursors(estate, astnode);

						if (rc == PLPGPSM_RC_UNDO)
						{
							RollbackAndReleaseCurrentSubTransaction();
							MemoryContextSwitchTo(oldCxt);
							CurrentResourceOwner = oldowner;

							if (!leave)
								rc = PLPGPSM_RC_OK;
						}
						else
						{
							ReleaseCurrentSubTransaction();
							MemoryContextSwitchTo(oldCxt);
							CurrentResourceOwner = oldowner;
						}

						SPI_restore_connection();
					}
					PG_CATCH();
					{
						PLpgPSM_condition c;
						ErrorData *edata;

						MemoryContextSwitchTo(oldCxt);
						edata = CopyErrorData();
						FlushErrorState();

						RollbackAndReleaseCurrentSubTransaction();
						MemoryContextSwitchTo(oldCxt);
						CurrentResourceOwner = oldowner;

						/* reconnect spi */
						SPI_restore_connection();

						c.astnode.type = PLPGPSM_CONDITION;
						c.type = PLPGPSM_CONDITION_SQLSTATE;
						c.sqlstate = edata->sqlerrcode;
						c.name = NULL;

						plpgpsm_eval_cleanup(estate);

						estate->handler = plpgpsm_search_handler((PLpgPSM_astnode *) stmt, &c);
						if (estate->handler == NULL)
							ReThrowError(edata);

						estate->edata = edata;
						push_edata(estate, estate->edata);

						if (estate->handler->astnode.parent != astnode)
						{
							/* outer handler */
							leave = true;
							estate->stop_node = estate->handler->astnode.parent;
							rc = PLPGPSM_RC_UNDO;
						}
						else
							leave = false;
					}
					PG_END_TRY();

					if (leave)
						return rc;

					/* UNDO handler call if exist */
					if (estate->handler != NULL)
					{
						PLpgPSM_astnode *stmt = estate->handler->stmt;
						ErrorData	*edata;
						int	rc;

						estate->handler = NULL;
						rc = eval_astnode(estate, (PLpgPSM_astnode *) stmt);

						edata = pop_edata(estate);
						free_edata(edata);
						pfree(edata);

						return rc;
					}
				}

				return PLPGPSM_RC_OK;
			}

		case PLPGPSM_STMT_LOOP:
		case PLPGPSM_STMT_WHILE:
		case PLPGPSM_STMT_REPEAT_UNTIL:
			{
				PLpgPSM_eb_stmt *stmt = (PLpgPSM_eb_stmt *) astnode;
				bool		leave;

				Assert (astnode->type != PLPGPSM_STMT_WHILE || stmt->expr != NULL);
				Assert (astnode->type != PLPGPSM_STMT_REPEAT_UNTIL || stmt->expr != NULL);
				Assert (astnode->type != PLPGPSM_STMT_LOOP || stmt->expr == NULL);

				for (;;)
				{
					if (astnode->type == PLPGPSM_STMT_WHILE && !plpgpsm_eval_boolean(estate, stmt->expr))
						break;

					rc = eval_stmts(estate, astnode, &leave);
					if (leave)
						return rc;

					if (astnode->type == PLPGPSM_STMT_REPEAT_UNTIL && plpgpsm_eval_boolean(estate, stmt->expr))
						break;
				}

				return PLPGPSM_RC_OK;
			}

		case PLPGPSM_STMT_LEAVE:
			{
				estate->stop_node = ((PLpgPSM_stmt_leave_iterate *) astnode)->stop_node;

				return PLPGPSM_RC_LEAVE;
			}

		case PLPGPSM_STMT_ITERATE:
			{
				estate->stop_node = ((PLpgPSM_stmt_leave_iterate *) astnode)->stop_node;

				return PLPGPSM_RC_ITERATE;
			}

		case PLPGPSM_STMT_RETURN:
			{
				PLpgPSM_stmt_return *stmt = (PLpgPSM_stmt_return *) astnode;

				if (stmt->expr != NULL)
				{
					estate->retval = plpgpsm_eval_datum(estate, stmt->expr, estate->rtinfo,
												    &estate->retisnull);
				}

				return PLPGPSM_RC_RETURN;
			}

		case PLPGPSM_STMT_PRINT:
			{
				StringInfoData		str;
				PLpgPSM_stmt_print *stmt = (PLpgPSM_stmt_print *) astnode;
				ListCell	*lc;
				bool	isfirst = true;

				initStringInfo(&str);
				foreach (lc, stmt->expr_list)
				{
					PLpgPSM_ESQL *esql = (PLpgPSM_ESQL *) lfirst(lc);
					char	*result;

					result = plpgpsm_eval_cstring(estate, esql);
					if (result == NULL)
						result = pstrdup("<NULL>");

					if (isfirst)
					{
						appendStringInfo(&str, "%s", result);
						isfirst = false;
					}
					else
						appendStringInfo(&str, " %s", result);

					pfree(result);
				}

				elog(NOTICE, "%s", str.data);
				pfree(str.data);

				return PLPGPSM_RC_OK;
			}

		case PLPGPSM_DECLARE_CURSOR:
			{
				PLpgPSM_declare_cursor *deccur = (PLpgPSM_declare_cursor *) astnode;
				PLpgPSM_cursor *cursor = deccur->cursor;
				PLpgPSM_datum *datum = &estate->datums[cursor->offset];

				declare_cursor_datum(cursor, datum);

				return PLPGPSM_RC_OK;
			}

		case PLPGPSM_STMT_OPEN:
			{
				PLpgPSM_stmt_open *stmt = (PLpgPSM_stmt_open *) astnode;
				PLpgPSM_referer *referer = stmt->cursor;
				PLpgPSM_cursor *cursor = (PLpgPSM_cursor *) referer->ref;

				open_cursor(estate, cursor);

				return PLPGPSM_RC_OK;
			}

		case PLPGPSM_STMT_CLOSE:
			{
				PLpgPSM_stmt_close *stmt = (PLpgPSM_stmt_close *) astnode;

				close_cursor(estate, (PLpgPSM_cursor *) stmt->cursor->ref, false);

				return PLPGPSM_RC_OK;
			}

		case PLPGPSM_STMT_FETCH:
			{
				PLpgPSM_stmt_fetch *stmt = (PLpgPSM_stmt_fetch *) astnode;
				PLpgPSM_referer *referer = stmt->cursor;
				PLpgPSM_cursor *cursor = (PLpgPSM_cursor *) referer->ref;
				PLpgPSM_datum *datum;
				PLpgPSM_fetch_extra *extra;

				Assert(cursor != NULL && cursor->astnode.type == PLPGPSM_CURSOR);
				datum = &estate->datums[cursor->offset];

				fetch_from_cursor(estate, cursor, datum);
				extra = (PLpgPSM_fetch_extra *) datum->extra;

				if (extra != NULL)
				{
					/* when prefetch is used */
					if (!extra->is_ready)
						elog(ERROR, "fetching from cursor is not prepared");

					plpgpsm_tuple_to_targets(estate, stmt->target_list,
										extra->cursor_tuple,
										extra->cursor_tupdesc);
				}
				else
				{
					/* prefetch is not used */
					plpgpsm_store_result_to_datums(estate, stmt->target_list);

					plpgpsm_eval_cleanup(estate);
				}

				if (estate->sqlcode == MAKE_SQLSTATE('0','2','0','0','0') && stmt->handler != NULL)
					return eval_handler(estate, stmt->handler);
				else
					return PLPGPSM_RC_OK;
			}

		case PLPGPSM_STMT_FOR:
			{
				PLpgPSM_eb_stmt *stmt = (PLpgPSM_eb_stmt *) astnode;
				PLpgPSM_datum *datum;
				bool	leave;

				Assert(stmt->cursor != NULL);

				datum = &estate->datums[stmt->cursor->offset];

				declare_cursor_datum(stmt->cursor, datum);
				open_cursor(estate, stmt->cursor);

				rc = PLPGPSM_RC_OK;

				for (;;)
				{
					fetch_from_cursor(estate, stmt->cursor, datum);
					if (estate->sqlcode == MAKE_SQLSTATE('0','2','0','0','0'))
						break;

					rc = eval_stmts(estate, astnode, &leave);

					if (leave)
						break;
				}

				close_cursor(estate, stmt->cursor, false);

				if (leave)
					return rc;

				return PLPGPSM_RC_OK;
			}

		case PLPGPSM_STMT_SIGNAL:
		case PLPGPSM_STMT_RESIGNAL:
			{
				PLpgPSM_stmt_signal *stmt = (PLpgPSM_stmt_signal *) astnode;
				char	*message = NULL;
				char	*detail = NULL;
				char	*hint = NULL;
				char	*condition_name = NULL;
				ListCell	*l;
				int			rc = PLPGPSM_RC_OK;
				int	sqlstate = 0;
				char		*substitute_message;

				if (stmt->astnode.type == PLPGPSM_STMT_RESIGNAL)
				{
					ErrorData	*edata;

					if (estate->stacked_edata == NULL)
						elog(ERROR, "there are no handled any signal yet");

					edata = estate->stacked_edata->edata;
					sqlstate = edata->sqlerrcode;

					if (edata->message != NULL)
						message = pstrdup(edata->message);
					if (edata->detail != NULL)
						detail = pstrdup(edata->detail);
					if (edata->hint != NULL)
						hint = pstrdup(edata->hint);

					/* ToDo: enhance ErrorData for this field */
					if (edata->detail_log != NULL)
						condition_name = pstrdup(edata->detail_log);

					if (stmt->condition == NULL)
						stmt->elog_level = edata->elevel;
				}

				foreach(l, stmt->signal_info_list)
				{
					PLpgPSM_signal_info *sinfo = (PLpgPSM_signal_info *) lfirst(l);
					char *str;

					Assert(sinfo->astnode.type == PLPGPSM_SIGNAL_INFO);

					str = plpgpsm_eval_cstring(estate, sinfo->expr);

					switch (sinfo->type)
					{
						case PLPGPSM_SIGNAL_INFO_MESSAGE:
							if (message != NULL)
								pfree(message);
							message = str;
							break;
						case PLPGPSM_SIGNAL_INFO_DETAIL:
							if (detail != NULL)
								pfree(detail);
							detail = str;
							break;
						case PLPGPSM_SIGNAL_INFO_HINT:
							if (hint != NULL)
								pfree(hint);
							hint = str;
							break;
					}
				}

				if (stmt->condition && stmt->condition->type == PLPGPSM_CONDITION_SQLSTATE)
				{
					sqlstate = stmt->condition->sqlstate;
					substitute_message = unpack_sql_state(sqlstate);
				}
				else if (stmt->condition && stmt->condition->type == PLPGPSM_CONDITION_NAMED)
				{
					sqlstate = stmt->condition->sqlstate;
					substitute_message = stmt->condition->name;

					if (condition_name != NULL)
						pfree(condition_name);
					condition_name = pstrdup(stmt->condition->name);

				}
				else
					substitute_message = unpack_sql_state(sqlstate);

				if (stmt->handler != NULL && stmt->handler->handler_type != PLPGPSM_HANDLER_UNDO)
				{
					ErrorData		edata;

					set_edata(&edata, stmt->elog_level, sqlstate,
										    message, detail, hint,
										    condition_name);
					estate->edata = &edata;
					estate->sqlcode = sqlstate;

					push_edata(estate, &edata);
					rc = eval_handler(estate, stmt->handler);
					pop_edata(estate);
					free_edata(&edata);

					return rc;
				}
				else if (stmt->handler != NULL && stmt->handler->handler_type == PLPGPSM_HANDLER_UNDO)
				{
					ErrorData *edata = (ErrorData *) palloc(sizeof(ErrorData));

					estate->edata = edata;
					push_edata(estate, edata);

					estate->sqlcode = sqlstate;
					set_edata(edata, stmt->elog_level, sqlstate,
										    message, detail, hint,
										    condition_name);
					estate->stop_node = stmt->handler->astnode.parent;
					estate->handler = stmt->handler;

					return PLPGPSM_RC_UNDO;
				}
				else
				{
					ereport(stmt->elog_level,
								(errcode(sqlstate),
								 errmsg_internal("%s", (message != NULL) ? message : substitute_message),
								 (detail != NULL) ? errdetail_internal("%s", detail) : 0,
								 (hint != NULL) ? errhint("%s", hint) : 0,
								 (condition_name != NULL) ? errdetail_log("%s", condition_name) : 0));
				}

				if (message != NULL)
					pfree(message);
				if (detail != NULL)
					pfree(detail);
				if (hint != NULL)
					pfree(hint);
				if (condition_name != NULL)
					pfree(condition_name);

				return rc;
			}

		case PLPGPSM_DECLARE_HANDLER:
		case PLPGPSM_DECLARE_CONDITION:
			break;

		case PLPGPSM_STMT_DIAGNOSTICS:
			{
				PLpgPSM_stmt_diagnostics *dstmt = (PLpgPSM_stmt_diagnostics *) astnode;
				ListCell	*l;
				ErrorData	*edata = NULL;

				if (dstmt->diag_area == PLPGPSM_DIAGAREA_CURRENT)
					edata = estate->edata;
				else
				{
					if (estate->stacked_edata != NULL)
						edata = estate->stacked_edata->edata;
				}

				foreach(l, dstmt->diag_info_list)
				{
					PLpgPSM_diag_info *dinfo = (PLpgPSM_diag_info *) lfirst(l);
					PLpgPSM_variable *var;
					PLpgPSM_datum *datum;
					Datum	value = (Datum) 0;
					Datum		cvalue;
					Oid	typeid = InvalidOid;

					var = (PLpgPSM_variable *) dinfo->target->ref;
					datum = &estate->datums[var->offset];

					switch (dinfo->type)
					{
						case PLPGPSM_DIAGINFO_RETURNED_SQLSTATE:
							if (edata != NULL)
								value = CStringGetTextDatum(unpack_sql_state(edata->sqlerrcode));
							else
								value = CStringGetTextDatum(unpack_sql_state(0));
							typeid = TEXTOID;
							break;

						case PLPGPSM_DIAGINFO_RETURNED_SQLCODE:
							if (edata != NULL)
								value = Int32GetDatum(edata->sqlerrcode);
							else
								value = Int32GetDatum(0);
							typeid = INT4OID;
							break;

						case PLPGPSM_DIAGINFO_MESSAGE_TEXT:
							if (edata != NULL)
								value = CStringGetTextDatum(edata->message ? edata->message : "");
							else
								value = CStringGetTextDatum("");
							typeid = TEXTOID;
							break;

						case PLPGPSM_DIAGINFO_DETAIL_TEXT:
							if (edata != NULL)
								value = CStringGetTextDatum(edata->detail ? edata->detail : "");
							else
								value = CStringGetTextDatum("");
							typeid = TEXTOID;
							break;

						case PLPGPSM_DIAGINFO_HINT_TEXT:
							if (edata != NULL)
								value = CStringGetTextDatum(edata->hint ? edata->hint : "");
							else
								value = CStringGetTextDatum("");

							typeid = TEXTOID;
							break;

						case PLPGPSM_DIAGINFO_ROW_COUNT:
							value = Int32GetDatum(estate->eval_processed);
							typeid = INT4OID;
							break;

						case PLPGPSM_DIAGINFO_CONDITION_IDENTIFIER:
							if (edata != NULL)
								value = CStringGetTextDatum(edata->detail_log ? edata->detail_log : "");
							else
								value = CStringGetTextDatum("");

							typeid = TEXTOID;
							break;
					}

					cvalue = plpgpsm_cast(estate, value, false, typeid, -1, datum->typeinfo);
					plpgpsm_set_datum(estate, datum, cvalue, false, false);

					if (typeid == TEXTOID && cvalue != value)
						pfree(DatumGetPointer(value));
				}

				return PLPGPSM_RC_OK;
			}

		default:
			elog(ERROR, "unsupported statement");
	}

	return PLPGPSM_RC_OK;
}

static int
eval_handler(PLpgPSM_execstate *estate, PLpgPSM_declare_handler *handler)
{
	int rc;

	rc = eval_astnode(estate, handler->stmt);
	if (rc != PLPGPSM_RC_OK)
		return rc;

	if (handler->handler_type == PLPGPSM_HANDLER_EXIT)
	{
		estate->stop_node = handler->astnode.parent;

		return PLPGPSM_RC_LEAVE;
	}

	return PLPGPSM_RC_OK;
}

static void
push_edata(PLpgPSM_execstate *estate, ErrorData *edata)
{
	PLpgPSM_stacked_ErrorData *e;

	e = (PLpgPSM_stacked_ErrorData *) palloc(sizeof(PLpgPSM_stacked_ErrorData));
	e->edata = edata;
	e->next = estate->stacked_edata;
	estate->stacked_edata = e;
}

static ErrorData *
pop_edata(PLpgPSM_execstate *estate)
{
	PLpgPSM_stacked_ErrorData *e = estate->stacked_edata;
	ErrorData   *result;

	Assert(e != NULL);

	result = e->edata;
	estate->stacked_edata = e->next;

	pfree(e);

	return result;
}
