--
-- Regression tests that were filed as cases in bug tickets,
-- referenced by bug number for historical interest.
--
SET client_min_messages TO NOTICE;

-- #2 --
SELECT '#2', ST_AsText(ST_Normalize(ST_Union(g)) ) FROM
( VALUES
('SRID=4326;MULTIPOLYGON(((1 1,1 2,2 2,2 1,1 1)))'),
('SRID=4326;MULTIPOLYGON(((2 1,3 1,3 2,2 2,2 1)))')
) AS v(g);

-- #11 --
SELECT '#11', ST_Distance (a.g, ST_Intersection(b.g, a.g)) AS distance
FROM (SELECT '01020000000200000050E8303FC2E85141B017CFC05A825541000000E0C0E85141000000205C825541'::geometry AS g) a,
	 (SELECT 'LINESTRING(4694792.35840419 5638508.89950758,4694793.20840419 5638506.34950758)'::geometry AS g) b;

-- #21 --
SELECT '#21', ST_AsEWKT(ST_LocateAlong(g, 4566)) FROM
( VALUES
(ST_GeomFromEWKT('SRID=31293;LINESTRINGM( 6193.76 5337404.95 4519, 6220.13 5337367.145 4566, 6240.889 5337337.383 4603 )'))
) AS v(g);

-- #22 --
SELECT ST_Within(g, 'POLYGON((0 0,10 0,20 10,10 20,0 20,-10 10,0 0))') FROM
(VALUES
('POLYGON((4 9,6 9,6 11,4 11,4 9))')
) AS v(g);

-- #33 --
CREATE TABLE road_pg (ID INTEGER, NAME VARCHAR(32));
SELECT '#33', AddGeometryColumn( '', 'public', 'road_pg','roads_geom', 330000, 'POINT', 2 );
DROP TABLE road_pg;

-- #44 --
SELECT '#44', ST_Relate(g1, g2, 'T12101212'), ST_Relate(g1, g2, 't12101212') FROM
(VALUES
('POLYGON((0 0, 2 0, 2 2, 0 2, 0 0))', 'POLYGON((1 1, 3 1, 3 3, 1 3, 1 1))')
) AS v(g1, g2);

-- #58 --
SELECT '#58', round(ST_xmin(g)),round(ST_ymin(g)),round(ST_xmax(g)),round(ST_ymax(g)) FROM (SELECT ST_Envelope('CIRCULARSTRING(220268.439465645 150415.359530563,220227.333322076 150505.561285879,220227.353105332 150406.434743975)'::geometry) as g)  AS foo;

-- #65 --
SELECT '#65', ST_AsGML(ST_GeometryFromText('CURVEPOLYGON(CIRCULARSTRING(4 2,3 -1.0,1 -1,-1.0 4,4 2))'));

-- #66 --
SELECT '#66', ST_AsText((ST_Dump(ST_GeomFromEWKT('CIRCULARSTRING(0 0,1 1,2 2)'))).geom);

-- #68 --
SELECT '#68a', ST_AsText(ST_ShiftLongitude(ST_GeomFromText('MULTIPOINT(1 3, 4 5)')));
SELECT '#68b', ST_AsText(ST_ShiftLongitude(ST_GeomFromText('CIRCULARSTRING(1 3, 4 5, 6 7)')));

-- #69 --
SELECT '#69', ST_AsText(ST_Translate(ST_GeomFromText('CIRCULARSTRING(220268 150415,220227 150505,220227 150406)'),1,2));

-- #70 --
SELECT '#70', ST_NPoints(ST_LinetoCurve(ST_Buffer('POINT(1 2)',3)));

-- #73 --
SELECT '#73', ST_AsText(ST_ForceCollection(ST_GeomFromEWKT('CIRCULARSTRING(1 1, 2 3, 4 5, 6 7, 5 6)')));

-- #80 --
SELECT '#80', ST_AsText(ST_Multi('MULTILINESTRING((0 0,1 1))'));

-- #83 --
SELECT '#83', ST_AsText(ST_Multi(ST_GeomFromText('CIRCULARSTRING(220268 150415,220227 150505,220227 150406)')));

-- #85 --
SELECT '#85', ST_Distance(ST_GeomFromText('CIRCULARSTRING(220268 150415,220227 150505,220227 150406)'), ST_Point(220268, 150415));

-- #112 --
SELECT '#112', ST_AsText(ST_CurveToLine('GEOMETRYCOLLECTION(POINT(-10 50))'::geometry));

-- #113 --
SELECT '#113', ST_LocateAlong('LINESTRING(0 0 0, 1 1 1)', 0.5);

-- #116 --
SELECT '#116', ST_AsText('010300000000000000');

-- #122 --
SELECT '#122', ST_AsText(ST_SnapToGrid(ST_GeomFromText('CIRCULARSTRING(220268 150415,220227 150505,220227 150406)'), 0.1));

-- #124 --
SELECT '#124a', ST_AsText(ST_GeomFromEWKT('COMPOUNDCURVE(CIRCULARSTRING(0 0,1 1,1 0),(1 0,30 5),CIRCULARSTRING(30 5,34 56,67 89))'));
SELECT '#124b', ST_AsText(ST_GeomFromEWKT('COMPOUNDCURVE(CIRCULARSTRING(0 0,1 1,1 0),(1 0,30 6),CIRCULARSTRING(30 5,34 56,67 89))'));

-- #145 --
SELECT '#145a', encode(ST_AsEWKB(ST_Buffer(ST_GeomFromText('LINESTRING(-116.93414544665981 34.16033385105459,-116.87777514700957 34.10831080544884,-116.86972224705954 34.086748622072776,-116.9327074288116 34.08458099517253,-117.00216369088065 34.130329331330216,-117.00216369088065 34.130329331330216)', 4326), 0),'ndr'),'hex');
SELECT '#145b', ST_Area(ST_Buffer(ST_GeomFromText('LINESTRING(-116.93414544665981 34.16033385105459,-116.87777514700957 34.10831080544884,-116.86972224705954 34.086748622072776,-116.9327074288116 34.08458099517253,-117.00216369088065 34.130329331330216,-117.00216369088065 34.130329331330216)', 4326), 0));

-- #146 --
SELECT '#146', ST_Distance(g1,g2), ST_Dwithin(g1,g2,0.01), ST_AsEWKT(g2) FROM (SELECT ST_geomFromEWKT('LINESTRING(1 2, 2 4)') As g1, ST_Collect(ST_GeomFromEWKT('LINESTRING(0 0, -1 -1)'), ST_GeomFromEWKT('MULTIPOINT(1 2,2 3)')) As g2) As foo;

-- #156 --
SELECT '#156', ST_AsEWKT('0106000000010000000103000000010000000700000024213D12AA7BFD40945FF42576511941676A32F9017BFD40B1D67BEA7E511941C3E3C640DB7DFD4026CE38F4EE531941C91289C5A7EFD40017B8518E3531941646F1599AB7DFD409627F1F0AE521941355EBA49547CFD407B14AEC74652194123213D12AA7BFD40945FF42576511941');

-- #157 --
SELECT
	'#157',
	ST_GeometryType(g) As newname,
	GeometryType(g) as oldname
FROM ( VALUES
	(ST_GeomFromText('POLYGON((-0.25 -1.25,-0.25 1.25,2.5 1.25,2.5 -1.25,-0.25 -1.25), (2.25 0,1.25 1,1.25 -1,2.25 0),(1 -1,1 1,0 0,1 -1))') ),
	( ST_Point(1,2) ),
	( ST_Buffer(ST_Point(1,2), 3) ),
	( ST_LineToCurve(ST_Buffer(ST_Point(1,2), 3)) ) ,
	( ST_LineToCurve(ST_Boundary(ST_Buffer(ST_Point(1,2), 3))) )
	) AS v(g);

-- #175 --
SELECT '#175', ST_AsEWKT(ST_GeomFromEWKT('SRID=26915;POINT(482020 4984378.)'));

-- #178 --
SELECT '#178a', ST_XMin(ST_MakeBox2D(ST_Point(5, 5), ST_Point(0, 0)));
SELECT '#178b', ST_XMax(ST_MakeBox2D(ST_Point(5, 5), ST_Point(0, 0)));

-- #179 --
SELECT '#179a', ST_MakeLine(ARRAY[NULL,NULL,NULL,NULL]);
SELECT '#179b', ST_MakeLine(ARRAY[NULL,NULL,NULL,NULL]);

-- #183 --
SELECT '#183', ST_AsText(ST_SnapToGrid(ST_LineToCurve(ST_LineMerge(ST_Collect(ST_CurveToLine(ST_GeomFromEWKT('CIRCULARSTRING(0 0, 1 1, 1 0)')),ST_GeomFromEWKT('LINESTRING(1 0, 0 1)') ))), 1E-10));

-- #210 --
SELECT '#210a', ST_Union(ARRAY[NULL,NULL,NULL,NULL]) ;
SELECT '#210b', ST_MakeLine(ARRAY[NULL,NULL,NULL,NULL]) ;

-- #213 --
SELECT '#213', round(ST_Perimeter(ST_CurveToLine(ST_GeomFromEWKT('CURVEPOLYGON(COMPOUNDCURVE(CIRCULARSTRING(0 0,2 0, 2 1, 2 3, 4 3),(4 3, 4 5, 1 4, 0 0)))'))));

-- #234 --
SELECT '#234', ST_AsText(ST_GeomFromText('COMPOUNDCURVE( (0 0,1 1) )'));

-- #239 --
--SELECT '#239', ST_AsSVG('010700002031BF0D0000000000');

-- #241 --
CREATE TABLE c ( the_geom GEOMETRY);
INSERT INTO c SELECT ST_MakeLine(ST_Point(-10,40),ST_Point(40,-10)) As the_geom;
INSERT INTO c SELECT ST_MakeLine(ST_Point(-10,40),ST_Point(40,-10)) As the_geom;
SELECT '#241', sum(ST_LineCrossingDirection(the_geom, ST_GeomFromText('LINESTRING(1 2,3 4)'))) FROM c;
DROP TABLE c;

-- #254 --
SELECT '#254', encode(ST_AsEWKB(ST_Segmentize(ST_GeomFromText('GEOMETRYCOLLECTION EMPTY'), 0.5),'ndr'),'hex');

-- #259 --
SELECT '#259', ST_Distance(ST_GeographyFromText('SRID=4326;POLYGON EMPTY'), ST_GeographyFromText('SRID=4326;POINT(1 2)'));

-- #260 --
SELECT '#260', round(ST_Distance(ST_GeographyFromText('SRID=4326;POINT(-10 40)'), ST_GeographyFromText('SRID=4326;POINT(-10 55)')));

-- #261 --
SELECT '#261', ST_Distance(ST_GeographyFromText('SRID=4326;POINT(-71.0325022849392 42.3793285830812)'),ST_GeographyFromText('SRID=4326;POLYGON((-71.0325022849392 42.3793285830812,-71.0325745928559 42.3793012556699,-71.0326708728343 42.3794450989722,-71.0326045866257 42.3794706688942,-71.0325022849392 42.3793285830812))'));

-- #262 --
SELECT '#262', ST_AsText(pt.the_geog) As wkt_pt, ST_Covers(poly.the_geog, pt.the_geog) As geog,
	ST_Covers(
		ST_Transform(CAST(poly.the_geog As geometry),32611),
		ST_Transform(CAST(pt.the_geog As geometry),32611)) As utm,
	ST_Covers(
		CAST(poly.the_geog As geometry),
		CAST(pt.the_geog As geometry)
	) As pca
FROM (SELECT ST_GeographyFromText('SRID=4326;POLYGON((-119.5434 34.9438,-119.5437 34.9445,-119.5452 34.9442,-119.5434 34.9438))') As the_geog)
	As poly
    CROSS JOIN
	(VALUES
		( ST_GeographyFromText('SRID=4326;POINT(-119.5434 34.9438)') ) ,
		( ST_GeographyFromText('SRID=4326;POINT(-119.5452 34.9442)') ) ,
		( ST_GeographyFromText('SRID=4326;POINT(-119.5434 34.9438)') ),
		( ST_GeographyFromText('SRID=4326;POINT(-119.5438 34.9443)')  )
	)
	As pt(the_geog);

-- #263 --
SELECT '#263', ST_AsEWKT(geometry(geography(pt.the_geom))) As wkt,
	ST_Covers(
		ST_Transform(poly.the_geom,32611),
		ST_Transform(pt.the_geom,32611)) As utm,
	ST_Covers(
		poly.the_geom,
		pt.the_geom)
	 As pca,
	ST_Covers(geometry(geography(poly.the_geom)),
		geometry(geography(pt.the_geom))) As gm_to_gg_gm_pca

FROM (SELECT ST_GeomFromEWKT('SRID=4326;POLYGON((-119.5434 34.9438,-119.5437 34.9445,-119.5452 34.9442,-119.5434 34.9438))') As the_geom)
	As poly
    CROSS JOIN
	(VALUES
		( ST_GeomFromEWKT('SRID=4326;POINT(-119.5434 34.9438)') ) ,
		( ST_GeomFromEWKT('SRID=4326;POINT(-119.5452 34.9442)') ) ,
		( ST_GeomFromEWKT('SRID=4326;POINT(-119.5434 34.9438)') ),
		( ST_GeomFromEWKT('SRID=4326;POINT(-119.5438 34.9443)')  )
	)
	As pt(the_geom);

-- #271 --
SELECT '#271', ST_Covers(
'POLYGON((-9.123456789 50,51 -11.123456789,-10.123456789 50,-9.123456789 50))'::geography,
'POINT(-10.123456789 50)'::geography
);

-- #272 --
SELECT '#272', ST_LineCrossingDirection(foo.line1, foo.line2) As l1_cross_l2 ,
    ST_LineCrossingDirection(foo.line2, foo.line1) As l2_cross_l1
FROM (SELECT
    ST_GeomFromText('LINESTRING(25 169,89 114,40 70,86 43)') As line1, ST_GeomFromText('LINESTRING(2.99 90.16,71 74,20 140,171 154)') As line2 ) As foo;

-- #277 --
SELECT '#277', ST_AsGML(2, ST_GeomFromText('POINT(1 1e308)'));

-- #299 --
SELECT '#299', round(ST_Y(geometry(ST_Intersection(ST_GeographyFromText('POINT(1.2456 2)'), ST_GeographyFromText('POINT(1.2456 2)')))));

-- #304 --

SELECT '#304';

CREATE OR REPLACE FUNCTION utmzone(geometry)
  RETURNS integer AS
$BODY$
DECLARE
    geomgeog geometry;
    zone int;
    pref int;

BEGIN
    geomgeog:= ST_Transform($1,4326);

    IF (ST_Y(geomgeog))>0 THEN
       pref:=32600;
    ELSE
       pref:=32700;
    END IF;

    zone:=floor((ST_X(geomgeog)+180)/6)+1;
    IF ( zone > 60 ) THEN zone := 60; END IF;

    RETURN zone+pref;
END;
$BODY$ LANGUAGE 'plpgsql' IMMUTABLE
  COST 100;

CREATE TABLE utm_dots ( the_geog geography, utm_srid integer);
INSERT INTO utm_dots SELECT geography(ST_SetSRID(ST_Point(i*10,j*10),4326)) As the_geog, utmzone(ST_SetSRID(ST_Point(i*10,j*10),4326)) As utm_srid FROM generate_series(-17,17) As i CROSS JOIN generate_series(-8,8) As j;

SELECT ST_AsText(the_geog) as the_pt,
       utm_srid,
       ST_Area(ST_Buffer(the_geog,10)) As the_area,
       ST_Area(geography(ST_Transform(ST_Buffer(ST_Transform(geometry(the_geog),utm_srid),10),4326))) As geog_utm_area
FROM utm_dots
WHERE ST_Area(ST_Buffer(the_geog,10)) NOT between 307 and 315
LIMIT 10;

SELECT '#304.a', Count(*) FROM utm_dots WHERE ST_DWithin(the_geog, 'POINT(0 0)'::geography, 3000000);

CREATE INDEX utm_dots_gix ON utm_dots USING GIST (the_geog);
SELECT '#304.b', Count(*) FROM utm_dots WHERE ST_DWithin(the_geog, 'POINT(0 0)'::geography, 300000);

DROP FUNCTION utmzone(geometry);
DROP TABLE utm_dots;

-- #408 --
SELECT '#408', substring(st_isvalidreason('0105000020E0670000010000000102000020E06700000100000016DA52BA62A04141FFF3AD290B735241') from E'.*\\[');
SELECT '#408.1', st_isvalid('01050000000100000001020000000100000000000000000000000000000000000000');
SELECT '#408.2', st_isvalidreason('01020000000100000000000000000000000000000000000000');
SELECT '#408.3', st_isvalid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
SELECT '#408.4', st_isvalidreason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

-- #457 --
SELECT '#457.1', st_astext(st_collectionExtract('POINT(0 0)', 1));
SELECT '#457.2', st_astext(st_collectionExtract('POINT(0 0)', 2));
SELECT '#457.3', st_astext(st_collectionExtract('POINT(0 0)', 3));
SELECT '#457.4', st_astext(st_collectionExtract('LINESTRING(0 0, 1 1)', 1));
SELECT '#457.5', st_astext(st_collectionExtract('LINESTRING(0 0, 1 1)', 2));
SELECT '#457.6', st_astext(st_collectionExtract('LINESTRING(0 0, 1 1)', 3));
SELECT '#457.7', st_astext(st_collectionExtract('POLYGON((0 0, 1 0, 1 1, 0 1, 0 0))', 1));
SELECT '#457.8', st_astext(st_collectionExtract('POLYGON((0 0, 1 0, 1 1, 0 1, 0 0))', 2));
SELECT '#457.9', st_astext(st_collectionExtract('POLYGON((0 0, 1 0, 1 1, 0 1, 0 0))', 3));

-- #835 --
SELECT '#835.1', st_astext(st_collectionExtract('POLYGON EMPTY', 1));
SELECT '#835.2', st_astext(st_collectionExtract('POLYGON EMPTY', 2));
SELECT '#835.3', st_astext(st_collectionExtract('POLYGON EMPTY', 3));
SELECT '#835.4', st_astext(st_collectionExtract('LINESTRING EMPTY', 1));
SELECT '#835.5', st_astext(st_collectionExtract('LINESTRING EMPTY', 2));
SELECT '#835.6', st_astext(st_collectionExtract('LINESTRING EMPTY', 3));
SELECT '#835.7', st_astext(st_collectionExtract('POINT EMPTY', 1));
SELECT '#835.8', st_astext(st_collectionExtract('POINT EMPTY', 2));
SELECT '#835.9', st_astext(st_collectionExtract('POINT EMPTY', 3));
SELECT '#835.10', st_astext(st_collectionExtract('GEOMETRYCOLLECTION EMPTY', 1));
SELECT '#835.11', st_astext(st_collectionExtract('GEOMETRYCOLLECTION EMPTY', 2));
SELECT '#835.12', st_astext(st_collectionExtract('GEOMETRYCOLLECTION EMPTY', 3));

-- #650 --
SELECT '#650', ST_AsText(ST_Collect(ARRAY[ST_MakePoint(0,0), ST_MakePoint(1,1), null, ST_MakePoint(2,2)]));

-- #662 --
--SELECT '#662', ST_MakePolygon(ST_AddPoint(ST_AddPoint(ST_MakeLine(ST_SetSRID(ST_MakePointM(i+m,j,m),4326),ST_SetSRID(ST_MakePointM(j+m,i-m,m),4326)),ST_SetSRID(ST_MakePointM(i,j,m),4326)),ST_SetSRID(ST_MakePointM(i+m,j,m),4326))) As the_geom FROM generate_series(-10,50,20) As i CROSS JOIN generate_series(50,70, 20) As j CROSS JOIN generate_series(1,2) As m ORDER BY i, j, m, i*j*m LIMIT 1;

-- #667 --
SELECT '#667', ST_AsText( ST_LineToCurve(ST_Buffer(ST_SetSRID(ST_Point(i,j),4326), j)), 4 ) As the_geom FROM generate_series(-10,50,10) As i CROSS JOIN generate_series(40,70, 20) As j ORDER BY i, j, i*j LIMIT 1;

-- #677 --
SELECT '#677.deprecated',round(ST_DistanceSpheroid(ST_GeomFromEWKT('MULTIPOLYGON(((-10 40,-10 55,-10 70,5 40,-10 40)))'), ST_GeomFromEWKT('MULTIPOINT(20 40,20 55,20 70,35 40,35 55,35 70,50 40,50 55,50 70)'), 'SPHEROID["GRS_1980",6378137,298.257222101]')) As result;
SELECT '#677',round(ST_DistanceSpheroid(ST_GeomFromEWKT('MULTIPOLYGON(((-10 40,-10 55,-10 70,5 40,-10 40)))'), ST_GeomFromEWKT('MULTIPOINT(20 40,20 55,20 70,35 40,35 55,35 70,50 40,50 55,50 70)'), 'SPHEROID["GRS_1980",6378137,298.257222101]')) As result;

-- #680 --
SELECT '#680', encode(ST_AsBinary(geography(foo1.the_geom),'ndr'),'hex') As result FROM ((SELECT ST_SetSRID(ST_MakePointM(i,j,m),4326) As the_geom FROM generate_series(-10,50,10) As i CROSS JOIN generate_series(50,70, 20) AS j CROSS JOIN generate_series(1,2) As m ORDER BY i, j, i*j*m)) As foo1 LIMIT 1;

-- #681 --
SELECT '#681a', ST_AsGML(ST_GeomFromText('POINT EMPTY', 4326));
SELECT '#681b', ST_AsGML(ST_GeomFromText('POLYGON EMPTY', 4326));
SELECT '#681c', ST_AsGML(ST_GeomFromText('LINESTRING EMPTY', 4326));
SELECT '#681d', ST_AsGML(ST_GeomFromText('MULTIPOINT EMPTY', 4326));
SELECT '#681e', ST_AsGML(ST_GeomFromText('MULTILINESTRING EMPTY', 4326));
SELECT '#681f', ST_AsGML(ST_GeomFromText('MULTIPOLYGON EMPTY', 4326));
SELECT '#681g', ST_AsGML(ST_GeomFromText('GEOMETRYCOLLECTION EMPTY', 4326));

-- #682 --
SELECT '#682', encode(ST_AsEWKB(ST_Buffer(ST_GeomFromText('POLYGON EMPTY',4326) , 0.5), 'ndr'),'hex');

-- #683 --
SELECT '#683', encode(ST_AsEWKB(ST_BuildArea(ST_GeomFromText('POINT EMPTY',4326)),'ndr'),'hex');

-- #684,#2109 --
SELECT '#684,#2109', ST_AsEWKT(ST_Centroid(ST_GeomFromText('POLYGON EMPTY',4326)));
SELECT '#2109', ST_AsEWKT(ST_Centroid(ST_GeomFromText('MULTILINESTRING ZM EMPTY',3395)));

-- #685 --
SELECT '#685', encode(ST_AsEWKB(ST_ConvexHull(ST_GeomFromText('POLYGON EMPTY',4326)),'ndr'),'hex');

-- #686 --
SELECT '#686', encode(ST_AsEWKB(ST_COLLECT(ST_GeomFromText('POLYGON EMPTY',4326),ST_GeomFromText('TRIANGLE EMPTY',4326)),'ndr'),'hex');

-- #687 --
SELECT '#687', ST_DFullyWithin(ST_GeomFromText('LINESTRING(-10 50,50 -10)',4326), ST_GeomFromText('POLYGON EMPTY',4326),5);

-- #689 --
SELECT '#689', ST_CoveredBy(ST_GeomFromText('POLYGON EMPTY'), ST_GeomFromText('LINESTRING(-10 50,50 -10)'));

-- #690 --
SELECT '#690';
SELECT encode(ST_AsEWKB(ST_MakeLine(ST_GeomFromText('POINT(-11.1111111 40)'), ST_GeomFromText('LINESTRING(-11.1111111 70,70 -11.1111111)')),'ndr'), 'hex') As result;

-- #693 --
SELECT '#693a', encode(ST_AsEWKB(ST_GeomFromEWKT('SRID=4326;POLYGONM((-71.1319 42.2503 1,-71.132 42.2502 3,-71.1323 42.2504 -2,-71.1322 42.2505 1,-71.1319 42.2503 0))'),'ndr'),'hex');
SELECT '#693b', encode(ST_AsEWKB(ST_GeomFromEWKT('SRID=4326;POLYGONM((-71.1319 42.2512 0,-71.1318 42.2511 20,-71.1317 42.2511 -20,-71.1317 42.251 5,-71.1317 42.2509 4,-71.132 42.2511 6,-71.1319 42.2512 30))'),'ndr'),'hex');

-- #694 --
SELECT '#694';
SELECT ST_MakePolygon('POINT(1 2)'::geometry);

-- #695 --
SELECT '#695';
SELECT ST_RemovePoint('POINT(-11.1111111 40)'::geometry, 1);

-- #696 --
SELECT '#696', encode(ST_AsEWKB(ST_Segmentize(ST_GeomFromEWKT('PolyhedralSurface( ((0 0 0, 0 0 1, 0 1 1, 0 1 0, 0 0 0)), ((0 0 0, 0 1 0, 1 1 0, 1 0 0, 0 0 0)), ((0 0 0, 1 0 0, 1 0 1, 0 0 1, 0 0 0)), ((1 1 0, 1 1 1, 1 0 1, 1 0 0, 1 1 0)), ((0 1 0, 0 1 1, 1 1 1, 1 1 0, 0 1 0)), ((0 0 1, 1 0 1, 1 1 1, 0 1 1, 0 0 1)) )'), 0.5),'ndr'),'hex');

-- #720 --
SELECT '#720', ST_AsText(ST_SnapTogrid(ST_Transform(ST_GeomFromText('MULTIPOINT(-10 40,-10 55,-10 70,5 40,5 55,5 70,20 40,20 55,20 70,35 40,35 55,35 70,50 40,50 55,50 70)',4326), 3395), 0.01));

-- #723 --
SELECT '#723',
 ST_IsEmpty(ST_SnapToGrid( ST_Intersection(a.geog, b.geog)::geometry, 0.00001))
FROM (VALUES (ST_GeogFromText('SRID=4326;POINT(-11.1111111 40)') ), (ST_GeogFromText('SRID=4326;POINT(-11.1111111 55)') ) ) As  a(geog) CROSS JOIN ( VALUES (ST_GeogFromText('SRID=4326;POINT(-11.1111111 40)') ), (ST_GeogFromText('SRID=4326;POINT(-11.1111111 55)') )) As b(geog);

-- #729 --
--SELECT '#729',ST_MakeLine(foo1.the_geom) As result FROM ((SELECT ST_GeomFromText('POINT EMPTY',4326) As the_geom UNION ALL SELECT ST_GeomFromText('MULTIPOINT EMPTY',4326) As the_geom UNION ALL SELECT ST_GeomFromText('MULTIPOLYGON EMPTY',4326) As the_geom UNION ALL SELECT ST_GeomFromText('LINESTRING EMPTY',4326) As the_geom UNION ALL SELECT ST_GeomFromText('MULTILINESTRING EMPTY',4326) As the_geom ) ) As foo1;

-- #804
SELECT '#804', ST_AsGML(3, 'SRID=4326;POINT(0 0)'::geometry, 0, 1);

-- #845
SELECT '#845', ST_Intersects('POINT(169.69960846592 -46.5061209281002)'::geometry, 'POLYGON((169.699607857174 -46.5061218662,169.699607857174 -46.5061195965597,169.699608806526 -46.5061195965597,169.699608806526 -46.5061218662,169.699607857174 -46.5061218662))'::geometry);

-- #834
SELECT '#834', ST_AsEWKT(ST_Normalize(ST_Force2D(ST_Intersection('LINESTRING(0 0,0 10,10 10,10 0)', 'LINESTRING(10 10 4,10 0 5,0 0 5)'))));

-- #884 --
CREATE TABLE foo (id integer, the_geom geometry);
INSERT INTO foo VALUES (1, st_geomfromtext('MULTIPOLYGON(((-113.6 35.4,-113.6 35.8,-113.2 35.8,-113.2 35.4,-113.6 35.4),(-113.5 35.5,-113.3 35.5,-113.3 35.7,-113.5 35.7,-113.5 35.5)))'));
INSERT INTO foo VALUES (2, st_geomfromtext('MULTIPOLYGON(((-113.7 35.3,-113.7 35.9,-113.1 35.9,-113.1 35.3,-113.7 35.3),(-113.6 35.4,-113.2 35.4,-113.2 35.8,-113.6 35.8,-113.6 35.4)),((-113.5 35.5,-113.5 35.7,-113.3 35.7,-113.3 35.5,-113.5 35.5)))'));

select '#884', id, ST_Within(
ST_GeomFromText('POINT (-113.4 35.6)'), the_geom
) from foo;

select '#938', 'POLYGON EMPTY'::geometry::box2d;

DROP TABLE foo;

-- #668 --
select '#668',box2d('CIRCULARSTRING(10 2,12 2,14 2)'::geometry) as b;

-- #711 --
select '#711', ST_GeoHash(ST_GeomFromText('POLYGON EMPTY',4326));

-- #712 --
SELECT '#712',ST_IsValid(ST_GeomFromText('POLYGON EMPTY',4326));

-- #756
WITH inp AS ( SELECT 'LINESTRING(0 0, 1 1)'::geometry as s,
                     'LINESTRING EMPTY'::geometry as e      )
 SELECT '#756.1', ST_Equals(s, st_multi(s)),
                  ST_Equals(s, st_collect(s, e))
 FROM inp;

-- #1023 --
select '#1023', 'POINT(10 4)'::geometry = 'POINT(10 4)'::geometry;
select '#1023.a', 'POINT(10 4)'::geometry = 'POINT(10 5)'::geometry;
select '#1023.b', postgis_addbbox('POINT(10 4)'::geometry) = 'POINT(10 4)'::geometry;

-- #1069 --
select '#1060', ST_Relate(ST_GeomFromText('POINT EMPTY',4326), ST_GeomFromText('POINT EMPTY',4326)) As result;

-- #1273 --
WITH p AS ( SELECT 'POINT(832694.188 816254.625)'::geometry as g )
SELECT '#1273', st_equals(p.g, postgis_addbbox(p.g)) from p;

-- Another for #1273 --
WITH p AS ( SELECT 'MULTIPOINT((832694.188 816254.625))'::geometry as g )
SELECT '#1273.1', st_equals(p.g, postgis_dropbbox(p.g)) from p;

-- #1292
SELECT '#1292', ST_AsText(ST_SnapToGrid(ST_GeomFromText(
	'GEOMETRYCOLLECTION(POINT(180 90),POLYGON((140 50,150 50,180 50,140 50),(140 60,150 60,180 60,140 60)))'
	, 4326), 0.00001)::geography);

-- #1292.1
SELECT '#1292.1', ST_AsText(ST_GeomFromText('POINT(180.00000000001 95)')::geography),
	ST_AsText(ST_GeomFromText('POINT(185 90.00000000001)')::geography);

-- #1320
SELECT '<#1320>';
CREATE TABLE A ( geom geometry(MultiPolygon, 4326),
                 geog geography(MultiPolygon, 4326) );
-- Valid inserts
INSERT INTO a(geog) VALUES('SRID=4326;MULTIPOLYGON (((0 0, 10 0, 10 10, 0 0)))'::geography);
INSERT INTO a(geom) VALUES('SRID=4326;MULTIPOLYGON (((0 0, 10 0, 10 10, 0 0)))'::geometry);
SELECT '#1320.geog.1', geometrytype(geog::geometry), st_srid(geog::geometry) FROM a where geog is not null;
SELECT '#1320.geom.1', geometrytype(geom), st_srid(geom) FROM a where geom is not null;
-- Type mismatches is not allowed
INSERT INTO a(geog) VALUES('SRID=4326;POLYGON ((0 0, 10 0, 10 10, 0 0))'::geography);
INSERT INTO a(geom) VALUES('SRID=4326;POLYGON ((0 0, 10 0, 10 10, 0 0))'::geometry);
SELECT '#1320.geog.2', geometrytype(geog::geometry), st_srid(geog::geometry) FROM a where geog is not null;
SELECT '#1320.geom.2', geometrytype(geom), st_srid(geom) FROM a where geom is not null;
-- Even if it's a trigger changing the type
CREATE OR REPLACE FUNCTION triga() RETURNS trigger AS
$$ BEGIN
	NEW.geom = ST_GeometryN(New.geom,1);
	NEW.geog = ST_GeometryN(New.geog::geometry,1)::geography;
	RETURN NEW;
END; $$ language plpgsql VOLATILE;
CREATE TRIGGER triga_before
  BEFORE INSERT ON a FOR EACH ROW
  EXECUTE PROCEDURE triga();
INSERT INTO a(geog) VALUES('SRID=4326;MULTIPOLYGON (((0 0, 10 0, 10 10, 0 0)))'::geography);
INSERT INTO a(geom) VALUES('SRID=4326;MULTIPOLYGON (((0 0, 10 0, 10 10, 0 0)))'::geometry);
SELECT '#1320.geog.3', geometrytype(geog::geometry), st_srid(geog::geometry) FROM a where geog is not null;
SELECT '#1320.geom.3', geometrytype(geom), st_srid(geom) FROM a where geom is not null;
DROP TABLE A;
DROP FUNCTION triga();
SELECT '</#1320>';

-- st_AsText POLYGON((0 0,10 0,10 10,0 0))

-- #1344
select '#1344', octet_length(ST_AsEWKB(st_makeline(g))) FROM ( values ('POINT(0 0)'::geometry ) ) as foo(g);

-- #1385
SELECT '#1385', ST_Extent(g) FROM ( select null::geometry as g ) as foo;

-- #657
SELECT '#657.1',Round(ST_X(ST_Project('POINT(175 10)'::geography, 2000000, 3.1415/2)::GEOMETRY)::numeric,2);
SELECT '#657.2',Round(ST_Distance(ST_Project('POINT(10 10)'::geography, 10, 0), 'POINT(10 10)'::geography)::numeric,2);
SELECT '#657.3',ST_DWithin(ST_Project('POINT(10 10)'::geography, 2000, pi()/2), 'POINT(10 10)'::geography, 2000);

-- #1305
SELECT '#1305.1',ST_AsText(ST_Project('POINT(10 10)'::geography, 0, 0));
WITH pts AS ( SELECT 'POINT(0 45)'::geography AS s, 'POINT(45 45)'::geography AS e )
SELECT '#1305.2',abs(ST_Distance(e, ST_Project(s, ST_Distance(s, e), ST_Azimuth(s, e)))) < 0.001 FROM pts;
SELECT '#1305.3',ST_Azimuth('POINT(0 45)'::geography, 'POINT(0 45)'::geography) IS NULL;

-- #1445
SELECT '01060000400200000001040000400100000001010000400000000000000000000000000000000000000000000000000101000040000000000000F03F000000000000F03F000000000000F03F'::geometry;
SELECT '01050000400200000001040000400100000001010000400000000000000000000000000000000000000000000000000101000040000000000000F03F000000000000F03F000000000000F03F'::geometry;
SELECT '01040000400200000001040000400100000001010000400000000000000000000000000000000000000000000000000101000040000000000000F03F000000000000F03F000000000000F03F'::geometry;
SELECT '01090000400200000001040000400100000001010000400000000000000000000000000000000000000000000000000101000040000000000000F03F000000000000F03F000000000000F03F'::geometry;
SELECT '010B0000400200000001040000400100000001010000400000000000000000000000000000000000000000000000000101000040000000000000F03F000000000000F03F000000000000F03F'::geometry;
SELECT '010C0000400200000001040000400100000001010000400000000000000000000000000000000000000000000000000101000040000000000000F03F000000000000F03F000000000000F03F'::geometry;

-- #1453
SELECT '#1453.1', ST_OrderingEquals('POINT EMPTY', 'POINT EMPTY');
SELECT '#1453.2', ST_OrderingEquals('POINT EMPTY', 'POINT Z EMPTY');

-- #1454
with inp as ( select 'MULTILINESTRING((0 0, 2 0))'::geometry as g )
SELECT '#1454', st_orderingequals(g,g) from inp;

-- #1414
SELECT '#1414', st_astext(st_Force3DZ('CURVEPOLYGON EMPTY'));

-- #1478
SELECT '#1478', encode(ST_AsEWKB('SRID=1;POINT EMPTY'::geometry::text::geometry,'ndr'),'hex');

-- #745
SELECT '#745', ST_AsEWKT(ST_Split('POLYGON((-72 42 1,-70 43 1,-71 41 1,-72 42 1))',
                                  'LINESTRING(-10 40 1,-9 41 1)'));

-- #1450
SELECT '#1450', GeometryType('POINT(0 0)'::geography), GeometryType('POLYGON EMPTY'::geography);

-- #1482
select '#1482', ST_Srid('POINT(0 0)'::geography(point, 0)::geometry);

-- #852
CREATE TABLE cacheable (id int, g geometry);
COPY cacheable FROM STDIN;
1	POINT(0.5 0.5000000000001)
2	POINT(0.5 0.5000000000001)
\.
select '#852.1', id, -- first run is not cached, consequent are cached
  st_intersects(g, 'POLYGON((0 0, 10 10, 1 0, 0 0))'::geometry),
  st_intersects(g, 'POLYGON((0 0, 1 1, 1 0, 0 0))'::geometry) from cacheable;
UPDATE cacheable SET g = 'POINT(0.5 0.5)';
-- New select, new cache
select '#852.2', id, -- first run is not cached, consequent are cached
  st_intersects(g, 'POLYGON((0 0, 10 10, 1 0, 0 0))'::geometry),
  st_intersects(g, 'POLYGON((0 0, 1 1, 1 0, 0 0))'::geometry) from cacheable;
DROP TABLE cacheable;

-- #1489
with inp AS ( SELECT
	st_multi('POINT EMPTY'::geometry) as mp,
	st_multi('LINESTRING EMPTY'::geometry) as ml,
	st_multi('POLYGON EMPTY'::geometry) as ma,
	st_multi('GEOMETRYCOLLECTION EMPTY'::geometry) as mm
) select '#1489',
	st_astext(mp), st_numgeometries(mp),
	st_astext(ml), st_numgeometries(ml),
	st_astext(ma), st_numgeometries(ma),
	st_astext(mm), st_numgeometries(mm)
FROM inp;

-- #1150
insert into spatial_ref_sys (srid, proj4text) values (500001,NULL);
insert into spatial_ref_sys (srid, proj4text) values (500002, '+proj=longlat +ellps=WGS84 +datum=WGS84 +no_defs');
select '#1150', st_astext(st_transform('SRID=500002;POINT(0 0)'::geometry,500001));
delete from spatial_ref_sys where srid in (500001,500002);

-- #1038
select '#1038', ST_AsSVG('POLYGON EMPTY'::geometry);

-- #1042
select '#1042',round((st_ymax(st_minimumboundingcircle('LINESTRING(-1 -1, 1 1)')) * st_xmax(st_minimumboundingcircle('LINESTRING(-1 -1, 1 1)')))::numeric,0);

-- #1170 --
SELECT '#1170', ST_Y(ST_Intersection( ST_GeogFromText( 'POINT(0 90)'), ST_GeogFromText( 'POINT(0 90)' ))::geometry);

-- #1264 --
SELECT '#1264', ST_DWithin('POLYGON((-10 -10, -10 10, 10 10, 10 -10, -10 -10))'::geography, 'POINT(0 0)'::geography, 0);

-- #1398
select '#1398a', st_astext(st_snaptogrid(st_project('POINT(-120 45)'::geography, 100000, radians(45))::geometry,0.000001));
select '#1398b', st_astext(st_snaptogrid(st_project('POINT(20 85)'::geography, 2000000, radians(0.1))::geometry,0.000001));

-- #1543
with inp as ( select
'0105000000020000000102000000040000000000000000000000000000000000000000000000000024400000000000000000000000000000244000000000000024400000000000000000000000000000000001020000000100000000000000000000000000000000000000'
::geometry as g )
select '#1543', st_astext(g), st_astext(st_buildarea(g)) from inp;

-- #1578
with inp as (
 select ST_Collect('POLYGON EMPTY', 'POLYGON EMPTY') as mp,
        'POINT(0 0)'::geometry as p
)
select '#1578', st_within(p, mp), st_intersects(p, mp) FROM inp;

-- #1580
select '#1580.1', ST_Summary(ST_Transform('SRID=4326;POINT(0 0)'::geometry, 3395));

do $$
declare
proj_version integer;
err_str text;
begin

    select ((regexp_matches(postgis_proj_version(), '(\d+)\.\d+'))[1])::integer into proj_version;
    select ST_Transform('SRID=4326;POINT(180 95)'::geometry, 3395); -- fails

exception
when others then
    err_str := SQLERRM;
    if proj_version >= 8 and SQLERRM = 'transform: Invalid coordinate (2049)' then
        raise notice '#1580.2: Caught a PROJ error';
    elsif proj_version < 8 and SQLERRM = 'transform: latitude or longitude exceeded limits (-14)' then
        raise notice '#1580.2: Caught a PROJ error';
    else
    	raise notice '#1580.2: Unexpected PROJ Result. Proj version = %. Error string: %', proj_version, err_str;
    end if;

end;
$$ language 'plpgsql';

select '#1580.3', ST_Summary(ST_Transform('SRID=4326;POINT(0 0)'::geometry, 3395));

-- #1596 --
CREATE TABLE road_pg (ID INTEGER, NAME VARCHAR(32));
SELECT '#1596.1', AddGeometryColumn( 'road_pg','roads_geom', 3395, 'POINT', 2 );
SELECT '#1596.2', UpdateGeometrySRID( 'road_pg','roads_geom', 330000);
SELECT '#1596.3', srid FROM geometry_columns
  WHERE f_table_name = 'road_pg' AND f_geometry_column = 'roads_geom';
SELECT '#1596.4', UpdateGeometrySRID( 'road_pg','roads_geom', 999000);
SELECT '#1596.5', srid FROM geometry_columns
  WHERE f_table_name = 'road_pg' AND f_geometry_column = 'roads_geom';
SELECT '#1596.6', UpdateGeometrySRID( 'road_pg','roads_geom', -1);
SELECT '#1596.7', srid FROM geometry_columns
  WHERE f_table_name = 'road_pg' AND f_geometry_column = 'roads_geom';
DROP TABLE road_pg;

-- #1596
WITH inp AS ( SELECT
 'POLYGON((-176 -22,-176 -21,-175 -21,-175 -22,-176 -22))'::geography as a,
 'POINT(-176 -22)'::geography as p
) SELECT '#1596', ST_Summary(ST_Intersection(a,p)) FROM inp;

-- #1695
SELECT '#1695', ST_AsEWKT(ST_SnapToGrid('MULTIPOLYGON(((0 0, 10 0, 10 10, 0 10, 0 0)))'::geometry, 20));

-- #1697 --
CREATE TABLE eg(g geography, gm geometry);
CREATE INDEX egi on eg using gist (g);
CREATE INDEX egind on eg using gist (gm gist_geometry_ops_nd);
INSERT INTO eg (g, gm)
 select 'POINT EMPTY'::geography,
        'POINT EMPTY'::geometry
 from generate_series(1,1024);
SELECT '#1697.1', count(*) FROM eg WHERE g && 'POINT(0 0)'::geography;
SELECT '#1697.2', count(*) FROM eg WHERE gm && 'POINT(0 0)'::geometry;
SELECT '#1697.3', count(*) FROM eg WHERE gm ~= 'POINT EMPTY'::geometry;
DROP TABLE eg;

-- #1734 --
create table eg (g geography);
create index egi on eg using gist (g);
INSERT INTO eg(g) VALUES (NULL);
INSERT INTO eg (g) VALUES ('POINT(0 0)'::geography);
INSERT INTO eg (g) select 'POINT(0 0)'::geography
       FROM generate_series(1,1024);
SELECT '#1734.1', count(*) FROM eg;
DROP table eg;

-- #1755 --
select '#1755', encode(ST_AsBinary(ST_GeographyFromText('SRID=4326;Point(85 35 0)'),'ndr'),'hex');

-- #1776 --
with inp as ( SELECT
 'POLYGON EMPTY'::geometry as A,
 'POLYGON((0 0, 10 0, 10 10, 0 0))'::geometry as B )
SELECT '#1776',
 ST_AsText(ST_SymDifference(A,B)), ST_AsText(ST_SymDifference(B, A))
FROM inp;

-- #1780 --
SELECT '#1780',ST_GeoHash('POINT(4 4)'::geometry) = ST_GeoHash('POINT(4 4)'::geography);

-- #1791 --
with inp as ( SELECT
 '010100000000000000004065C0041AD965BE5554C0'::geometry as a,
 '010100000001000000004065C0041AD965BE5554C0'::geometry as b
) SELECT '#1791', round(ST_Azimuth(a,b)*10)/10 from inp;

-- #1799 --
SELECT '#1799', ST_Segmentize('LINESTRING(0 0, 10 0)'::geometry, 0);

-- #1936 --
select st_astext(st_geomfromgml(
    '<gml:Polygon xmlns:gml="http://www.opengis.net/gml/3.2"
    gml:id="HPA.15449990010" srsName="urn:ogc:def:crs:EPSG::4326"
    srsDimension="2">
    <gml:exterior>
    <gml:Ring>
    <gml:curveMember>
    <gml:LineString gml:id="HPA.15449990010.1">
    <gml:posList>711540.35 1070163.61 711523.82 1070166.54 711521.30 1070164.14 711519.52 1070162.44 711518.57 1070164.62 712154.47 1070824.94</gml:posList>
    </gml:LineString>
    </gml:curveMember>
    <gml:curveMember>
    <gml:Curve gml:id="HPA.15449990010.2">
    <gml:segments><gml:ArcString>
    <gml:posList>712154.47 1070824.94 712154.98 1070826.04 712154.41 1070827.22</gml:posList>
    </gml:ArcString>
    </gml:segments>
    </gml:Curve>
    </gml:curveMember>
    <gml:curveMember>
    <gml:LineString gml:id="HPA.15449990010.3">
    <gml:posList>712154.41 1070827.22 712160.31 1070837.07 712160.92 1070835.36 712207.89 1071007.95</gml:posList>
    </gml:LineString>
    </gml:curveMember>
    <gml:curveMember>
    <gml:Curve gml:id="HPA.15449990010.4"><gml:segments><gml:ArcString><gml:posList>712207.89 1071007.95 712207.48 1071005.59 712208.38 1071001.28</gml:posList></gml:ArcString></gml:segments></gml:Curve></gml:curveMember><gml:curveMember><gml:LineString gml:id="HPA.15449990010.5"><gml:posList>712208.38 1071001.28 712228.74 1070949.67 712233.98 1070936.15 712124.93 1070788.72</gml:posList></gml:LineString></gml:curveMember><gml:curveMember><gml:Curve gml:id="HPA.15449990010.6"><gml:segments><gml:ArcString><gml:posList>712124.93 1070788.72 712124.28 1070785.87 712124.63 1070783.38</gml:posList></gml:ArcString></gml:segments></gml:Curve></gml:curveMember><gml:curveMember><gml:LineString gml:id="HPA.15449990010.7"><gml:posList>712124.63 1070783.38 712141.04 1070764.12 712146.60 1070757.01 711540.35 1070163.61</gml:posList></gml:LineString></gml:curveMember></gml:Ring></gml:exterior>
    <gml:interior>
    <gml:LinearRing>
    <gml:posList>713061.62 1070354.46 713053.59 1070335.12 713049.58 1070315.92 713049.65 1070298.33 713061.62 1070354.46</gml:posList>
    </gml:LinearRing>
    </gml:interior>
    </gml:Polygon>'));

-- #1957 --
SELECT '#1957', ST_Distance(ST_Makeline(ARRAY['POINT(1 0)'::geometry]), 'POINT(0 0)'::geometry);

-- #1978 --
SELECT '#1978', round(ST_Length(ST_GeomFromText('CIRCULARSTRING(0 0,1 0,0 0)',0))::numeric,4);

-- #1996 --
SELECT '#1996', ST_AsGeoJSON(ST_GeomFromText('POINT EMPTY'));

-- #2001 --
SELECT '#2001', ST_AsText(ST_CurveToLine(ST_GeomFromText('CURVEPOLYGON((0 0, 0 1, 1 1, 0 0))'), 2));

-- #2028 --
SELECT '#2028', ST_AsText(ST_Multi('TRIANGLE((0 0, 0 1, 1 1, 0 0))'));

-- #2035 START ------------------------------------------------------------

-- Simple geographic table, with single point.
CREATE TABLE "city" (
    "id" integer,
    "name" varchar(30) NOT NULL,
    "point" geometry(POINT,4326) NOT NULL
);
CREATE INDEX "city_point_id" ON "city" USING GIST ( "point" );

-- Initial data, with points around the world.
INSERT INTO "city" (id, name, point) VALUES (1, 'Houston', 'SRID=4326;POINT(-95.363151 29.763374)');
INSERT INTO "city" (id, name, point) VALUES (2, 'Dallas', 'SRID=4326;POINT(-95.363151 29.763374)');
INSERT INTO "city" (id, name, point) VALUES (3, 'Oklahoma City', 'SRID=4326;POINT(-97.521157 34.464642)');
INSERT INTO "city" (id, name, point) VALUES (4, 'Wellington', 'SRID=4326;POINT(174.783117 -41.315268)');
INSERT INTO "city" (id, name, point) VALUES (5, 'Pueblo', 'SRID=4326;POINT(-104.609252 38.255001)');
INSERT INTO "city" (id, name, point) VALUES (6, 'Lawrence', 'SRID=4326;POINT(-95.23506 38.971823)');
INSERT INTO "city" (id, name, point) VALUES (7, 'Chicago', 'SRID=4326;POINT(-87.650175 41.850385)');
INSERT INTO "city" (id, name, point) VALUES (8, 'Victoria', 'SRID=4326;POINT(-123.305196 48.462611)');

-- This query, or COUNT(*), does not return anything; should return 6 cities,
-- excluding Pueblo and Victoria.  The Polygon is a simple approximation of
-- Colorado.
SELECT '#2035a', Count(*) FROM "city"
  WHERE "city"."point" >> ST_GeomFromEWKT('SRID=4326;POLYGON ((-109.060253 36.992426, -109.060253 41.003444, -102.041524 41.003444, -102.041524 36.992426, -109.060253 36.992426))');

-- However, when a LIMIT is placed on statement, the query suddenly works.
SELECT '#2035b', Count(*) FROM "city"
  WHERE "city"."point" >> ST_GeomFromEWKT('SRID=4326;POLYGON ((-109.060253 36.992426, -109.060253 41.003444, -102.041524 41.003444, -102.041524 36.992426, -109.060253 36.992426))') LIMIT 6;

DROP TABLE "city";
-- #2035 END --------------------------------------------------------------

-- #2084 --
SELECT '#2084', num, ST_Within('POINT(-54.394 56.522)', "the_geom"), ST_CoveredBy('POINT(-54.394 56.522)', "the_geom")
FROM ( VALUES
(1, '0103000000010000000E00000051C6F7C5A5324BC02EB69F8CF13F4C40F12EA4C343364BC0326AA2CF47434C402BC1A8A44E364BC02A50E10852434C407F2990D959364BC0A0D1730B5D434C404102452C62364BC0ECF335CB65434C400903232F6B364BC0F635E84B6F434C40BD0CC51D6F364BC0D2805EB873434C40B9E6E26F7B364BC0F20B93A982434C40D9FAAF73D3344BC0FE84D04197444C40BD5C8AABCA344BC0CED05CA791444C4023F2237EC5344BC02A84F23E8E444C40BDCDD8077B324BC0C60FB90F01434C409FD1702E65324BC04EF1915C17404C4051C6F7C5A5324BC02EB69F8CF13F4C40'::geometry),
(2, '0103000000010000001C00000003F25650F73B4BC098477F523E3E4C40C9A6A344CE3C4BC0C69698653E3E4C40BDD0E979373E4BC0081FA0FB723E4C400FD252793B3E4BC01A137F14753E4C40537170E998414BC070D3BCE314414C4023FC51D499474BC0D4D100DE024F4C40638C47A984454BC024130D52F0504C40B9442DCDAD404BC03A29E96168554C40C7108DEE20404BC07C7C26FBE7554C40195D6BEF533F4BC0E20391459A564C40239FE40E9B344BC08C1ADB6B41514C40132D3F7095314BC0BA2ADF33124F4C409DB91457952D4BC02C7B681F2B4C4C4089DC60A8C32C4BC07C5C3810924B4C40D7ED409DF22A4BC0F64389963C4A4C405D1EF818AC2A4BC00EC84274084A4C401B48A46DFC294BC0B271A8DF85494C40E78AA6B393294BC01ED0EFFB37494C4081C64B3789294BC0DC5BE7DF2E494C409B23329287294BC0F0D6974E2D494C40CD22D5D687294BC0844316D72C494C40F5229D4FE2294BC002F19825AB484C40A3D0BD5AE9294BC06C0776A9A2484C409FD1702E65324BC04EF1915C17404C409F860AA7BD324BC0162CA390E33F4C40539A5C1C23334BC0FE86B04EB03F4C4081511DFF90334BC088FF36D4873F4C4003F25650F73B4BC098477F523E3E4C40'::geometry),
(3, '010300000001000000100000008D57CD101A214BC0AECDD34E072C4C400DBB72E6EC274BC0A8088D60E32C4C40CF8FD7E6734E4BC0B22695BE4A324C40BFA74213934F4BC020BE505D4C354C4057CD4BEE454E4BC0BA6CF3940F3D4C40E7BDC5FD263E4BC09A4B297D5B484C4073A46A86701C4BC0B287F08D93364C4045501F86701C4BC05EBDB78D93364C40A37DB6586D1C4BC0841E7D2891364C409FBF445F6D1C4BC01E225C5690364C40D1BA97726D1C4BC06E2AF7EA8D364C4019B60C9B751C4BC0D2FD702575364C40FDE4394B5E1F4BC08C40F231CC2F4C402343DF40F51F4BC022008E3D7B2E4C400BB57B45F9204BC0908CE2EA3A2C4C408D57CD101A214BC0AECDD34E072C4C40'::geometry)
) AS f(num, the_geom);

-- #2112 -- Start
SELECT '#2112a', ST_3DDistance(a,b), ST_ASEWKT(ST_3DShortestLine(a,b))
FROM (SELECT 'POLYGON((1 1 1, 5 1 1,5 5 1, 1 5 1,1 1 1))'::geometry as a, 'LINESTRING(0 0 2, 0 0 0,5 5 2)'::geometry as b
     ) as foo;

SELECT '#2112b', ST_3DDistance(a,b), ST_ASEWKT(ST_3DShortestLine(a,b))
FROM (SELECT 'POLYGON((1 1 1, 5 1 1,5 5 1, 1 5 1,1 1 1))'::geometry as a, 'LINESTRING(1 0 2, 1 0 0,5 5 -1)'::geometry as b
     ) as foo;
-- 2112 -- End

SELECT '#2108', ST_AsEWKT(ST_LineInterpolatePoint('SRID=3395;LINESTRING M EMPTY'::geometry, 0.5));
SELECT '#2117', ST_AsEWKT(ST_PointOnSurface('SRID=3395;MULTIPOLYGON M EMPTY'::geometry));

SELECT '#2110.1', 'POINT(0 0)'::geometry = 'POINT EMPTY'::geometry;
SELECT '#2110.2', 'POINT EMPTY'::geometry = 'POINT EMPTY'::geometry;
SELECT '#2110.3', 'POINT(0 0)'::geometry = 'POINT(0 0)'::geometry;

SELECT '#2145',
round(ST_Length(St_Segmentize(ST_GeographyFromText('LINESTRING(-89.3000030518 28.2000007629,-89.1999969482 89.1999969482,-89.1999969482 89.1999969482)'), 10000))::numeric,0);

SELECT '#2232', ST_AsSVG('LINESTRING(0 0, 0.4 0, 0.8 0, 1.2 0,1.6 0, 2 0)'::geometry,1,0);

-- #2307 --
SELECT '#2307', ST_AsText(ST_Normalize(ST_MakeValid('0106000020E6100000010000000103000000010000000A0000004B7DA956B99844C0DB0790FE8B4D1DC010BA74A9AF9444C049AFFC5B8C4D1DC03FC6CC690D9844C0DD67E5628C4D1DC07117B56B0D9844C0C80ABA67C45E1DC0839166ABAF9444C0387D4568C45E1DC010BA74A9AF9444C049AFFC5B8C4D1DC040C3CD74169444C0362EC0608C4D1DC07C1A3B77169444C0DC3ADB40B2641DC03AAE5F68B99844C0242948DEB1641DC04B7DA956B99844C0DB0790FE8B4D1DC0'::geometry)),4);

SELECT '#2409', ST_Summary('GEOMETRYCOLLECTION(MULTISURFACE(POLYGON((0 0, 0 1, 1 1, 1 0, 0 0)),CURVEPOLYGON(COMPOUNDCURVE((0 0, 0 1, 1 1, 1 0, 0 0),CIRCULARSTRING(0 0, 0 1, 1 1, 1 2, 0 0)))),TIN(((0 0, 0 1, 1 1, 0 0))))'::geometry);

SELECT '#2415.1', ST_AsText(ST_Multi(
  'COMPOUNDCURVE((0 0, 10 0),CIRCULARSTRING(10 0, 15 1, 20 10))'
));
SELECT '#2415.2', ST_AsText(ST_Multi(
  'CURVEPOLYGON(CIRCULARSTRING(10 0,15 1,20 0,18 5,20 10,10 10,10 0))'
));

SELECT '#2412', ST_AsText(ST_LineToCurve('LINESTRING(0 0,10 0,20 0)'));

SELECT '#2420.1', ST_AsText(ST_LineToCurve('LINESTRING(0 0,10 0,10 10,0 10,0 0)'));
SELECT '#2420.2', ST_AsText(ST_LineToCurve('LINESTRING(0 0,10 0,10 10,0 10)'));

SELECT '#2423', ST_AsText(ST_SnapToGrid(ST_CurveToLine(ST_LineToCurve(
    ST_Normalize(ST_Intersection(ST_Buffer(ST_Point(0,0),10),ST_MakeEnvelope(-10,0,10,10)))
), 4), 0.0001), 3);

SELECT '#2424', ST_AsText(ST_SnapToGrid(ST_CurveToLine(
  'MULTICURVE(COMPOUNDCURVE((0 0, 10 0),CIRCULARSTRING(10 0, 20 1, 30 10)))',
2),1));

SELECT '#2427', st_astext(st_pointn(ST_CurveToLine('CIRCULARSTRING(-1 0,0 1,0 -1)'),1));

SELECT '#2168',  ST_Distance(g1,g2)::numeric(16,8)  As dist_g1_g2, ST_Distance(g2,g1)::numeric(16,8) AS dist_g2_g1,ST_Distance(g1,g2) - ST_Distance(g2,g1)
  FROM (SELECT 'POINT(18.5107234 54.7587757)'::geography As g1, 'POINT(18.58218 54.7344227)'::geography As g2) As a;

-- #2556 --

CREATE TABLE images (id integer, name varchar, extent geography(POLYGON,4326));
INSERT INTO images VALUES (47409, 'TDX-1_2010-10-06T19_44_2375085', 'SRID=4326;POLYGON((-59.4139571913088 82.9486103943668,-57.3528882462655 83.1123152898828,-50.2302874208478 81.3740574826097,-51.977353304689 81.2431047148532,-59.4139571913088 82.9486103943668))'::geography);
INSERT INTO images VALUES (1, 'first_image', 'SRID=4326;POLYGON((-162.211667 88.046667,-151.190278 87.248889,-44.266389 74.887778,-40.793889 75.043333,-162.211667 88.046667))'::geography);
SELECT '#2556' AS ticket, id, round(ST_Distance(extent, 'SRID=4326;POLYGON((-46.625977 81.634149,-46.625977 81.348076,-48.999023 81.348076,-48.999023 81.634149,-46.625977 81.634149))'::geography)) from images;
DROP TABLE images;

-- #2692
WITH v AS ( SELECT 'CIRCULARSTRING(0 0, 1 1, 2 2)'::geometry AS g FROM generate_series(1,3) )
SELECT '#2692a', ST_AsText(st_collect(g)) FROM v;
WITH v AS ( SELECT 'COMPOUNDCURVE((0 0, 1 1), CIRCULARSTRING(1 1, 1 2, 3 2))'::geometry AS g FROM generate_series(1,3) )
SELECT '#2692b', ST_AsText(st_collect(g)) FROM v;
WITH v AS ( SELECT 'TRIANGLE((0 0, 1 1, 1 0, 0 0))'::geometry AS g FROM generate_series(1,3) )
SELECT '#2692c', ST_AsText(st_collect(g)) FROM v;

SELECT '#2704', ST_AsText(ST_GeomFromGML('<?xml version="1.0"?>
<gml:Polygon xmlns:gml="http://www.opengis.net/gml/3.2"
             xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance"
             xsi:schemaLocation="http://www.opengis.net/gml/3.2 http://schemas.opengis.net/gml/3.2.1/gml.xsd"
             gml:id="ID1">
    <gml:exterior>
        <gml:LinearRing>
            <gml:pos>0 0</gml:pos>
            <gml:pos>0 1</gml:pos>
            <gml:pos>1 1</gml:pos>
            <gml:pos>1 0</gml:pos>
            <gml:pos>0 0</gml:pos>
        </gml:LinearRing>
    </gml:exterior>
</gml:Polygon>'));

SELECT '#2712', ST_AsText(ST_Segmentize('LINESTRING EMPTY'::geometry, 0.5));

SELECT '#2717',
   ST_AsText(ST_StartPoint(g)),
   ST_AsText(ST_EndPoint(g)),
   ST_AsText(ST_PointN(g, 1)),
   ST_AsText(ST_PointN(g, 2)),
   ST_AsText(ST_PointN(g, 3)),
   ST_AsText(ST_PointN(g, 4)),
   ST_AsText(ST_PointN(g, 5))
   FROM (
     SELECT 'COMPOUNDCURVE((-1 -1, 1 1), CIRCULARSTRING(1 1, 2 2, 3 1))'::geometry AS g
   ) AS foo;

SELECT '#2788', valid, reason, ST_AsText(location) from ST_IsValidDetail('POLYGON((0 0, 0 1, 2 1, 2 2, 1 2, 1 0, 0 0))'::geometry);

SELECT '#2870', ST_Summary('Point(151.215289 -33.856885)'::geometry::bytea::geography) ;

SELECT '#2956', st_astwkb(null,0) is null;

-- #2985 --
CREATE TABLE test_analyze_crash (a integer not null, g geometry);
INSERT INTO test_analyze_crash values (1, '0102000020E6100000010000006D1092A47FF33440AD4ECD9B00334A40');
ANALYZE test_analyze_crash;
SELECT '#2985', ST_AsText(g) FROM test_analyze_crash;
DROP TABLE test_analyze_crash;

-- #2996 --
WITH
  input AS (SELECT 'SRID=4326;POLYGON((26426 65078,26531 65242,26075 65136,26096 65427,26426 65078))'::geometry AS geom),
  mbc   AS (SELECT (mb).center, (mb).radius FROM (SELECT ST_MinimumBoundingRadius(geom) mb FROM input) sq)
SELECT '#2996', radius::numeric(8,2), ST_Length(ST_LongestLine(geom, center))::numeric(8,2) FROM input, mbc;

-- #3119 --
SELECT '#3119a', floor(ST_LengthSpheroid('SRID=4326;LINESTRING (-72.640965 42.11867, -72.6395 42.1187)', 'SPHEROID["GRS_1980",6378137,298.257222101]'));
-- polygons are also handled
SELECT '#3119b', floor(ST_LengthSpheroid('SRID=4326;POLYGON ((-72.640965 42.11867, -72.6395 42.1187, -72.64065 42.11819, -72.640965 42.11867))', 'SPHEROID["GRS_1980",6378137,298.257222101]'));
-- for polygons with holes, all rings are considered
SELECT '#3119c', floor(ST_LengthSpheroid('SRID=4326;POLYGON ((-72.640965 42.11819, -72.640965 42.1187, -72.6395 42.1187, -72.6395 42.11819, -72.640965 42.11819),
          (-72.63965 42.118305, -72.64083 42.118305, -72.64083 42.118595, -72.63965 42.118595, -72.63965 42.118305))', 'SPHEROID["GRS_1980",6378137,298.257222101]'));
-- triangles also work
SELECT '#3119d', floor(ST_LengthSpheroid('SRID=4326;TRIANGLE ((-72.640965 42.11867, -72.6395 42.1187, -72.64065 42.11819, -72.640965 42.11867))', 'SPHEROID["GRS_1980",6378137,298.257222101]'));

SELECT '#3172', ST_AsText(ST_AddMeasure('LINESTRING(0 0,0 0)', 1, 2));

--SELECT '#3244a', ST_AsText(ST_3DClosestPoint('POINT(0 0 0)', 'POINT(0 0)'));
--SELECT '#3244b', ST_AsText(ST_3DClosestPoint('POINT(0 0)', 'POINT(0 0 0)'));

SELECT '#3300', ST_AsText(ST_SnapToGrid(Box2D('CURVEPOLYGON(CIRCULARSTRING(-71.0821 42.3036,-71.4821 42.3036,-71.7821 42.7036,-71.0821 42.7036,-71.0821 42.3036),(-71.1821 42.4036,-71.3821 42.6036,-71.3821 42.4036,-71.1821 42.4036))'::Geometry)::geometry,0.0001));

SELECT '#3355',  ST_Intersects(
         'LINESTRING(124.983539 1.419224,91.181596 29.647798)'::geography
       , ST_Segmentize('LINESTRING(124.983539 1.419224,91.181596 29.647798)'::geography, 47487290)::geography);

SELECT '#3356', ST_Summary(wkt::geometry) As wkt_geom,
   ST_Summary(wkt::geography) As wkt_geog,
   ST_Summary(wkt::geometry::geography) As geom_geog
FROM (VALUES (
     'LINESTRING(124.983539 1.419224,91.181596 29.647798)'::text ),
 ('LINESTRING(124.983539 1.419224,91.181596 29.647798, 91.28 29.647)'::text  ) ) As f(wkt)
ORDER BY wkt;

SELECT '#3367', ST_AsText(ST_RemoveRepeatedPoints('POLYGON EMPTY'::geometry));

SELECT '#3368', ST_AsTWKB('0106000000010000000103000000010000001F0000007CCD1788E93A68C1B264ED35E160594198825A8FEA3A68C198868434E16059413A4E5BD5EB3A68C11A92C936E1605941225C7746FA3A68C147BCA921E1605941E49C3DE8123B68C184E964FDE0605941FE87AA821A3B68C1F50EFBF1E0605941C2BA16591A3B68C11312C4C0E2605941E8ACC5C8193B68C1848C95B6E9605941AF164E57193B68C1E4175CB5F0605941ADE0D204193B68C1030DF2BAF7605941CA6056DC183B68C13C4554B9FC605941FCA886C3183B68C1062253B901615941E89667BA183B68C139D028BA06615941E62E78B7183B68C1612D445F1F615941B3321071183B68C14FECFED82C6159410789236D183B68C165EF39352D615941B7C22A65183B68C134B466902D615941F7D33559183B68C16725AFB130615941CFA6FEE9163B68C111D8661631615941E999DE9B163B68C15112D66D31615941DE3D404A163B68C1C87B53B73161594193A6C1F5153B68C1FEBD50F2316159416E7A069F153B68C16D985B1E32615941C8B4B646153B68C1B0BD1E3B3261594148607DED143B68C17B78624832615941C9E5BF020F3B68C153352CBF32615941AAE3C27FEA3A68C183B40EAE3261594119EB767CE93A68C1B4F989A932615941C8120C81E73A68C164B064B1326159411F66AF89E73A68C1836D233EE16059417CCD1788E93A68C1B264ED35E1605941'::geometry,3) AS geom;

SELECT '#3375', ST_AsText(ST_RemoveRepeatedPoints('GEOMETRYCOLLECTION(POINT(0 -7))'::geometry, 1000));

-- #3399
WITH g as (
select 'POLYGON((1 0, 0 1, 1 2, 2 1, 1 0))'::geometry as geom
),
n as (
select n from unnest(ARRAY[-1,0,1,10,100,1000]) n
),
pts as (
  select n,(st_dump(st_generatepoints(geom, n))).geom from g,n
)
select '#3399' as t, n, count(*) from
g, pts
where st_intersects(g.geom, pts.geom)
group by n
ORDER BY n;

-- #3461
SELECT '#3461', ST_GeomFromKML('<Polygon>
                <innerBoundaryIs>
                    <LinearRing>
                        <coordinates>15.4314185364794,47.07424936394101,0 15.4319999913023,47.0745907424833,0 15.4304554154307,47.0762201181668,0 15.4297898819917,47.0762268569417,0 15.4290787803951,47.0767510408816,0 15.428814076225,47.07705890737439,0 15.4283323302523,47.0781601391252,0 15.4280670755831,47.0781119179139,0 15.428345820955,47.0773709566434,0 15.428377720989,47.0769281206226,0 15.4287760037479,47.0764644755462,0 15.429245707349,47.07569497547091,0 15.4314185364794,47.07424936394101,0</coordinates>
                    </LinearRing>
                </innerBoundaryIs>
            </Polygon>');

-- #3437
WITH
mp AS (SELECT ST_Collect(ST_MakePoint(-c, c*c)) AS geom FROM generate_series(1, 5) c),
p  AS (SELECT (ST_Dump(geom)).geom FROM mp)
SELECT '#3437a' AS t, count(*) FROM mp INNER JOIN p ON ST_Intersects(mp.geom, p.geom)
UNION ALL
SELECT '#3437b' AS t, count(*) FROM mp INNER JOIN p ON ST_Contains(mp.geom, p.geom)
UNION ALL
SELECT '#3437c' AS t, count(*) FROM mp INNER JOIN p ON ST_ContainsProperly(mp.geom, p.geom)
UNION ALL
SELECT '#3437d' AS t, count(*) FROM mp INNER JOIN p ON ST_Covers(mp.geom, p.geom)
UNION ALL
SELECT '#3437e' AS t, count(*) FROM mp INNER JOIN p ON ST_Within(p.geom, mp.geom);

-- #3470
SELECT '#3470', ST_Polygonize(ARRAY[NULL]::geometry[]) IS NULL;
SELECT '#3470b', ST_Area(ST_Polygonize(ARRAY[NULL, 'LINESTRING (0 0, 10 0, 10 10)', NULL, 'LINESTRING (0 0, 10 10)', NULL]::geometry[]));

-- #3569
WITH clustr AS (
      SELECT ST_Polygonize(ST_MakeEnvelope(1, 2, 3, 4)) AS geom
    )
    SELECT '#3569', ST_CollectionHomogenize(geom)::box2d
    FROM clustr;

-- #3578
SELECT '#3578a', ST_NumInteriorRings('POLYGON EMPTY');
SELECT '#3578b', ST_NumInteriorRings('CURVEPOLYGON EMPTY');

-- #3579
with
        params as (
        select
            11 :: float as sidewalk_offset,
            1 :: float  as epsilon
    ),
        road as (
-- L-shaped road, 10 m
        select 'SRID=3857;LINESTRING(10 0, 0 0, 0 10)' :: geometry as geom
    ),
        sidewalks as (
        select ST_Collect(
                   ST_OffsetCurve(geom, sidewalk_offset),
                   ST_OffsetCurve(geom, -sidewalk_offset)
               ) geom
        from road, params
    )
select
    '#3579', ST_Intersects(road.geom, sidewalks.geom),
-- should be false
    ST_Intersects(ST_Buffer(road.geom, sidewalk_offset + epsilon), sidewalks.geom) -- should be true
from road, sidewalks, params;

-- #3620
SELECT '#3620a', ST_AsText(ST_MinimumBoundingCircle('POINT (3 7)'));
SELECT '#3620b', ST_AsText(ST_MinimumBoundingCircle('LINESTRING (2 8, 2 8)'));

-- #3627
SELECT '#3627a', ST_AsEncodedPolyline('SRID=4326;LINESTRING(-0.250691 49.283048,-0.250633 49.283376,-0.250502 49.283972,-0.251245 49.284028,-0.251938 49.284232,-0.251938 49.2842)', 6);
SELECT '#3627b', ST_Equals(geom, ST_LineFromEncodedPolyline(ST_AsEncodedPolyline(geom, 7), 7)) FROM (VALUES ('SRID=4326;LINESTRING (0 0, 1 1)')) AS v (geom);

-- #3704
SELECT '#3704', ST_AsX3D('LINESTRING EMPTY') = '';

-- #3709
select '#3709', ST_SnapToGrid(ST_Project('SRID=4326;POINT(1 1)'::geography, 100000, 20)::geometry, 0.0001) = ST_SnapToGrid(ST_Project('SRID=4326;POINT(1 1)'::geography, -100000, 20+pi())::geometry, 0.0001);

-- #3719
select '#3719a', ST_IsValid('CURVEPOLYGON((25495445.625 6671632.625, 25495445.625 6671711.375, 25495555.375 6671711.375, 25495555.375 6671632.625, 25495445.625 6671632.625), COMPOUNDCURVE(CIRCULARSTRING(25495368.0441 6671726.9312,25495368.3959388 6671726.93601515,25495368.7478 6671726.9333), (25495368.7478 6671726.9333,25495368.0441 6671726.9312)))');
select '#3719b', ST_IsValid('CURVEPOLYGON(COMPOUNDCURVE(CIRCULARSTRING(1.0441 2.9312,1.3959388 2.93601515,1.7478 2.9333), (1.7478 2.9333,1.0441 2.9312)))'::geometry);

-- #3774
select '#3774', abs(pi() + 2 - st_length('COMPOUNDCURVE(CIRCULARSTRING(0 0, 1 1, 2 0), (2 0, 4 0))'::geometry)) < 0.000000001;

-- #1014
SELECT '#1014a', ST_AsText(g) FROM (
	SELECT 'POINT(-0 0)'::geometry AS g
	UNION
	SELECT 'POINT(0 0)'::geometry AS g
) a;
SELECT '#1014b', ST_AsText(g) FROM (
	SELECT 'POINT(0 1)'::geometry AS g
	UNION
	SELECT 'POINT(0 1)'::geometry AS g
) a;
CREATE TABLE rec (id integer, g geometry);
INSERT INTO rec VALUES (1, 'POINT(0 1)');
INSERT INTO rec VALUES (2, 'POINT(1 2)');
INSERT INTO rec VALUES (3, 'POINT(2 3)');
WITH RECURSIVE path (id, g) AS (
    SELECT id, g FROM rec WHERE id = 1
    UNION
    SELECT rec.id, rec.g
    FROM path, rec
    WHERE ST_Y(path.g) = ST_X(rec.g)
)
SELECT '#1014c', id, st_astext(g) FROM path;
SELECT '#1014d', ST_AsEWKT(g) as t FROM (
	SELECT 'SRID=1;POINT(0 1)'::geometry AS g
	UNION
	SELECT 'SRID=2;POINT(0 1)'::geometry AS g
) a ORDER BY t;
DROP TABLE IF EXISTS rec;

-- #3930
SELECT '#3930', round(1000*st_area(st_minimumboundingcircle('MULTIPOLYGON (((-76.5442733764648 34.5878295898438, -76.5551452636719 34.6106605529785, -76.5377502441406 34.6139221191406, -76.4844818115234 34.6976318359375, -76.4322967529297 34.760684967041, -76.3746795654297 34.8128700256348, -76.2877044677734 34.877010345459, -76.4692611694336 34.6932830810547, -76.5442733764648 34.5878295898438)), ((-76.0170059204102 35.0694351196289, -75.9756927490234 35.1161842346191, -75.8539352416992 35.1672821044922, -75.7637023925781 35.1922836303711, -75.8115386962891 35.1640205383301, -75.9039459228516 35.1324920654297, -76.0028762817383 35.0694351196289, -76.0170059204102 35.0694351196289)), ((-76.0271682739258 36.5567169189453, -75.998664855957 36.5566520690918, -75.9119186401367 36.5425300598145, -75.9248046875 36.4739761352539, -75.9772796630859 36.4780158996582, -75.9762878417969 36.5179252624512, -76.0271682739258 36.5567169189453)), ((-75.4912185668945 35.6704978942871, -75.5336227416992 35.768856048584, -75.4569778442383 35.6173973083496, -75.5262985229492 35.2279167175293, -75.7492904663086 35.189826965332, -75.6915664672852 35.2349891662598, -75.521484375 35.2813568115234, -75.4754180908203 35.5644950866699, -75.4912185668945 35.6704978942871)), ((-83.9379959106445 34.9893913269043, -83.9885482788086 34.9890670776367, -84.1182403564453 34.988224029541, -84.3238525390625 34.9890098571777, -84.2910385131836 35.2105445861816, -84.225944519043 35.261604309082, -84.1797256469727 35.2409896850586, -84.1016464233398 35.2455520629883, -84.0427780151367 35.272575378418, -84.0308609008789 35.2925224304199, -84.0292053222656 35.3252906799316, -84.0063095092773 35.372859954834, -84.0126495361328 35.4076232910156, -83.954704284668 35.4554595947266, -83.9100112915039 35.4764785766602, -83.8812255859375 35.5105857849121, -83.8302001953125 35.5190620422363, -83.77587890625 35.552604675293, -83.6728744506836 35.5649719238281, -83.6138610839844 35.5717391967773, -83.56103515625 35.55517578125, -83.5057983398438 35.5595512390137, -83.4582901000977 35.5972785949707, -83.387092590332 35.6252174377441, -83.3430252075195 35.6532592773438, -83.2984161376953 35.6563262939453, -83.2591247558594 35.6910095214844, -83.253303527832 35.7007064819336, -83.2438507080078 35.7182159423828, -83.1853485107422 35.7288856506348, -83.1436614990234 35.7626838684082, -83.1181869506836 35.7638092041016, -83.0599594116211 35.7825775146484, -82.9870071411133 35.7739906311035, -82.9627532958984 35.7918510437012, -82.9068222045898 35.8722152709961, -82.9140701293945 35.9278678894043, -82.8959732055664 35.9483604431152, -82.8562698364258 35.9474258422852, -82.8086700439453 35.9208717346191, -82.7764434814453 35.9565734863281, -82.7735977172852 35.9875030517578, -82.7632293701172 35.9995460510254, -82.6438903808594 36.0517234802246, -82.628044128418 36.0543403625488, -82.6044006347656 36.0429878234863, -82.5922317504883 36.0224494934082, -82.6058044433594 36.003547668457, -82.5993041992188 35.9632987976074, -82.5541458129883 35.9561080932617, -82.5069351196289 35.972541809082, -82.4751968383789 35.9931755065918, -82.4084243774414 36.0753173828125, -82.3738555908203 36.0986976623535, -82.3119277954102 36.1221504211426, -82.2623062133789 36.1203765869141, -82.2077331542969 36.1470146179199, -82.154052734375 36.1396217346191, -82.1180801391602 36.0962562561035, -82.0777587890625 36.1001396179199, -82.0204544067383 36.129711151123, -81.9331130981445 36.2633209228516, -81.9111557006836 36.2907524108887, -81.8305740356445 36.3346557617188, -81.7304916381836 36.3293418884277, -81.7094573974609 36.3337249755859, -81.7403793334961 36.3618583679199, -81.7410736083984 36.3917846679688, -81.6982803344727 36.4717788696289, -81.7027969360352 36.5193405151367, -81.6699981689453 36.5896492004395, -81.3452987670898 36.5728645324707, -80.9034423828125 36.5652122497559, -80.8381576538086 36.5634384155273, -80.6110534667969 36.5572967529297, -80.4353103637695 36.5510444641113, -80.048095703125 36.5471343994141, -80.024055480957 36.5450248718262, -79.7174453735352 36.5478897094727, -79.5102996826172 36.5476570129395, -79.2170639038086 36.5497817993164, -79.1443252563477 36.5460586547852, -78.7966995239258 36.5435333251953, -78.7373886108398 36.5460739135742, -78.4588088989258 36.5414810180664, -78.3212509155273 36.5455322265625, -78.0516662597656 36.5524749755859, -77.8988571166992 36.5529441833496, -77.7639312744141 36.5534400939941, -77.3200531005859 36.5539169311523, -77.1773529052734 36.5562858581543, -76.9241333007812 36.5541458129883, -76.921630859375 36.5541572570801, -76.5635833740234 36.5552520751953, -76.49755859375 36.5558128356934, -76.3302536010742 36.5560569763184, -76.1273956298828 36.5571632385254, -76.0459594726562 36.5569534301758, -76.0332107543945 36.5143737792969, -76.091064453125 36.5035667419434, -75.97607421875 36.4362144470215, -75.9697647094727 36.4151191711426, -76.0016098022461 36.4189147949219, -75.9512557983398 36.3654708862305, -75.9281234741211 36.4232444763184, -75.9245910644531 36.3509483337402, -75.8000564575195 36.1128158569336, -75.7988510131836 36.0728187561035, -75.8551635742188 36.1056671142578, -75.9137649536133 36.244800567627, -75.9575119018555 36.2594528198242, -75.9419326782227 36.2943382263184, -76.0089721679688 36.3195953369141, -75.9571838378906 36.1937713623047, -75.9813385009766 36.1697273254395, -76.1831665039062 36.3152389526367, -76.2189025878906 36.2966079711914, -76.1127090454102 36.1744194030762, -76.1419372558594 36.1476898193359, -76.234977722168 36.1633605957031, -76.2989273071289 36.2142295837402, -76.275505065918 36.1103706359863, -76.4805297851562 36.0797920227051, -76.4204254150391 36.0586051940918, -76.5230102539062 36.0071678161621, -76.5940017700195 36.0101318359375, -76.6490173339844 36.0657081604004, -76.6332092285156 36.0371170043945, -76.6901550292969 36.0496101379395, -76.7265090942383 36.1568222045898, -76.6887359619141 36.2945175170898, -76.7766418457031 36.3583297729492, -76.9240798950195 36.3924446105957, -76.7413482666016 36.3151664733887, -76.7074966430664 36.2661323547363, -76.7450637817383 36.2339172363281, -76.7606735229492 36.1445922851562, -76.6937637329102 35.9929695129395, -76.7411270141602 35.9366264343262, -76.6972198486328 35.9415435791016, -76.4094696044922 35.977466583252, -76.3714828491211 35.9323425292969, -76.2137680053711 35.9768753051758, -76.0896377563477 35.9629135131836, -76.0260467529297 35.9204254150391, -76.0759124755859 35.7568016052246, -76.0430679321289 35.6838493347168, -76.1673049926758 35.6968421936035, -76.1047439575195 35.6636390686035, -76.0212097167969 35.6690940856934, -75.9878540039062 35.892707824707, -75.8180541992188 35.9235191345215, -75.7489624023438 35.8693389892578, -75.7293701171875 35.6651725769043, -75.779052734375 35.578685760498, -75.8914947509766 35.6312675476074, -76.052978515625 35.4146041870117, -76.1816177368164 35.3415336608887, -76.4988021850586 35.4162178039551, -76.5337524414062 35.4501991271973, -76.4584350585938 35.5044250488281, -76.4465255737305 35.5508651733398, -76.5189437866211 35.5776443481445, -76.4925384521484 35.5417861938477, -76.6381988525391 35.520336151123, -76.6287689208984 35.4378967285156, -76.7053756713867 35.4119338989258, -77.1037673950195 35.5501861572266, -76.9831848144531 35.4365005493164, -76.6949005126953 35.3504257202148, -76.6145172119141 35.2729187011719, -76.5912094116211 35.3120803833008, -76.5409774780273 35.3036880493164, -76.5071258544922 35.2487678527832, -76.6400909423828 35.1725196838379, -76.605224609375 35.1385612487793, -76.6242218017578 35.0643501281738, -76.6780090332031 35.0240859985352, -76.8494644165039 34.9822196960449, -76.9414901733398 35.0274276733398, -76.943244934082 35.0700340270996, -77.0478820800781 35.092155456543, -77.1046752929688 35.0879859924316, -77.1105880737305 35.0660362243652, -77.0011901855469 35.0528869628906, -76.9131088256836 34.9364700317383, -76.6470489501953 34.9063339233398, -76.6700592041016 34.9699974060059, -76.5297775268555 34.9746360778809, -76.4450302124023 35.016658782959, -76.4240570068359 34.9462471008301, -76.364013671875 34.9431114196777, -76.3294067382812 34.9759559631348, -76.3147125244141 34.9488143920898, -76.4699020385742 34.7850608825684, -76.6253128051758 34.7197532653809, -77.0505294799805 34.698917388916, -77.1489562988281 34.764331817627, -77.12939453125 34.6849136352539, -77.1566009521484 34.6606369018555, -77.5386352539062 34.4570083618164, -77.5867614746094 34.4209365844727, -77.6097259521484 34.4350433349609, -77.6026153564453 34.4126281738281, -77.7505264282227 34.305046081543, -77.864387512207 34.1927375793457, -77.894401550293 34.0691795349121, -77.9267578125 34.0620346069336, -77.9607315063477 34.1892433166504, -77.9585266113281 33.9925804138184, -78.0348052978516 33.9142913818359, -78.579719543457 33.8819923400879, -78.6557159423828 33.948673248291, -79.0745010375977 34.304573059082, -79.4471435546875 34.6190605163574, -79.4559707641602 34.634090423584, -79.6675033569336 34.8006629943848, -79.6859588623047 34.8052558898926, -79.9199523925781 34.807918548584, -80.3252792358398 34.8147621154785, -80.5615463256836 34.8152275085449, -80.8000640869141 34.8161125183105, -80.7856521606445 34.9406394958496, -80.8401641845703 35.0020179748535, -80.8947143554688 35.0597343444824, -80.9277954101562 35.1012496948242, -81.0398864746094 35.0372009277344, -81.0655517578125 35.0664825439453, -81.0284423828125 35.1054077148438, -81.0490417480469 35.1320114135742, -81.0493011474609 35.1515312194824, -81.3228225708008 35.1637573242188, -81.362174987793 35.1628532409668, -81.7653579711914 35.1824722290039, -81.8705902099609 35.1831169128418, -81.9714431762695 35.1882820129395, -82.2101745605469 35.1931266784668, -82.2783279418945 35.1950073242188, -82.3207702636719 35.1841888427734, -82.3508605957031 35.1926727294922, -82.3601226806641 35.1829490661621, -82.3713760375977 35.1827239990234, -82.3896102905273 35.2082405090332, -82.4379196166992 35.1695594787598, -82.4667434692383 35.1735000610352, -82.5246353149414 35.1545600891113, -82.5700302124023 35.1494903564453, -82.6544952392578 35.119457244873, -82.6860504150391 35.1214637756348, -82.6880340576172 35.0978012084961, -82.6973571777344 35.0912322998047, -82.7713470458984 35.0854225158691, -82.8876953125 35.0553703308105, -83.0072784423828 35.0242042541504, -83.1062850952148 35.0002784729004, -83.5130081176758 34.9920234680176, -83.5494155883789 34.9895362854004, -83.9379959106445 34.9893913269043)), ((-75.7731552124023 36.2292556762695, -75.5449676513672 35.7883605957031, -75.7027359008789 36.049861907959, -75.7408676147461 36.0503234863281, -75.7831726074219 36.2251930236816, -75.901985168457 36.5561981201172, -75.8781661987305 36.5558738708496, -75.7731552124023 36.2292556762695)))')));

-- #4011
SELECT '#4011',
ST_GeometryType(ST_LineMerge(geom)) AS linemerge,
ST_AsText(geom) AS geom,
ST_IsEmpty(geom) AS empty,
ST_IsValid(geom) AS valid
FROM (VALUES
('LINESTRING(0 0, 1 1)'),
('MULTILINESTRING((0 0, 1 1), (1 1, 2 2))'),
('MULTILINESTRING((0 0, 1 1), EMPTY)'),
('MULTILINESTRING(EMPTY, EMPTY)'),
(NULL),
('POLYGON((0 0, 1 0, 1 1, 0 0))'),
('GEOMETRYCOLLECTION(LINESTRING(0 0, 1 1))'),
('MULTILINESTRING EMPTY'),
('MULTILINESTRING((0 0, 0 0))')
) as f(geom);

-- #4025
select '#4025', ST_DistanceCPA (
	'LINESTRING M (
	3002640.82925984 5833770.09678249 1180471354.482,
	3003084.56741608 5833866.22147656 1180471411.791,
	3003325.8165722 5833942.66399188 1180471442.914,
	3003634.9521919 5834039.57644606 1180471482.436,
	3003701.57229565 5833872.50949516 1180471494.55)'::geometry,
	'LINESTRING M (
	2996311.19255982 5844593.73486637 1180471513.163,
	2996310.72190852 5844712.60937002 1180471523.37,
	2996311.81673704 5844916.90864168 1180471541.171)'::geometry
	);

-- a butterfly polygon in all cases
SELECT '#4037.1', ST_AsText(ST_Intersection('POLYGON((0 0, 10 10, 0 10, 10 0, 0 0))', ST_MakeEnvelope(2,2,8,8)));
SELECT '#4037.2', ST_AsText(ST_Difference('POLYGON((0 0, 10 10, 0 10, 10 0, 0 0))', ST_MakeEnvelope(2,2,8,8)));
SELECT '#4037.3', ST_AsText(ST_SymDifference('POLYGON((0 0, 10 10, 0 10, 10 0, 0 0))', ST_MakeEnvelope(2,2,8,8)));
SELECT '#4037.4', ST_AsText(ST_Union('POLYGON((0 0, 10 10, 0 10, 10 0, 0 0))', ST_MakeEnvelope(2,2,8,8)));

-- #4055
SELECT '#4055a', ST_SRID(unnest(ST_ClusterWithin(ARRAY['SRID=4326;POINT (3 7)'::geometry, 'SRID=4326;LINESTRING (3 0, 3 9)'], 0)));
SELECT '#4055b', ST_SRID(unnest(ST_ClusterIntersecting(ARRAY['SRID=4326;POINT (3 7)'::geometry, 'SRID=4326;LINESTRING (3 0, 3 9)'])));

--#4089
select '#4089', st_astext(st_geomfromtwkb(st_AsTWKB(st_GeometryFromText('LINESTRING Z(1 1 1, 3 3 1)'), 1, 0, 0, false, true)));

-- #4081
select '#4081',
ST_DWithin(a, b, 110575, true) as within_110575_true,
ST_DWithin(a, b, 110576, true) as within_110576_true
from (
    select ST_GeogFromText('SRID=4326;POINT(1.0 2.0)') as a, ST_GeogFromText('SRID=4326;POINT(1.0 1.0)') as b
) as points;

-- #4144
DROP TABLE IF EXISTS bug_4144_table;
CREATE TABLE bug_4144_table (
  geom geometry NOT NULL DEFAULT NULL
);

INSERT INTO bug_4144_table (geom)
  VALUES ('GEOMETRYCOLLECTION(POINT(-3.385894e+38 0 0),POINT(0 0 0))');

ANALYZE bug_4144_table;
DROP TABLE IF EXISTS bug_4144_table;

--4295
-- CIP - collection within polygon
SELECT 'contains210', ST_contains('POLYGON((0 0, 0 10, 10 10, 10 0, 0 0))'::geometry, 'GEOMETRYCOLLECTION (POINT(5 5),LINESTRING(1 1,2 2))'::geometry);
-- CIP - collection on edge of polygon
SELECT 'contains211', ST_contains('POLYGON((0 0, 0 10, 10 10, 10 0, 0 0))'::geometry, 'GEOMETRYCOLLECTION (POINT(10 10), LINESTRING(0 0, 0 10))'::geometry);
-- CIP - collection outside polygon
SELECT 'contains212', ST_contains('POLYGON((0 0, 0 10, 10 10, 10 0, 0 0))'::geometry, 'GEOMETRYCOLLECTION (POINT(-1 0),LINESTRING(11 5,12 5))'::geometry);
-- CIP - elements of the collection fully outside and fully inside polygon
SELECT 'contains213', ST_contains('POLYGON((0 0, 0 10, 10 10, 10 0, 0 0))'::geometry, 'GEOMETRYCOLLECTION (POINT(-1 0), LINESTRING(4 4,5 5))'::geometry);

-- CIP - collection within polygon
SELECT 'within210', ST_within('POLYGON((0 0, 0 10, 10 10, 10 0, 0 0))'::geometry, 'GEOMETRYCOLLECTION (POINT(5 5),LINESTRING(1 1,2 2))'::geometry);
-- CIP - collection on edge of polygon
SELECT 'within211', ST_within('POLYGON((0 0, 0 10, 10 10, 10 0, 0 0))'::geometry, 'GEOMETRYCOLLECTION (POINT(10 10), LINESTRING(0 0, 0 10))'::geometry);
-- CIP - collection outside polygon
SELECT 'within212', ST_within('POLYGON((0 0, 0 10, 10 10, 10 0, 0 0))'::geometry, 'GEOMETRYCOLLECTION (POINT(-1 0),LINESTRING(11 5,12 5))'::geometry);
-- CIP - elements of the collection fully outside and fully inside polygon
SELECT 'within213', ST_within('POLYGON((0 0, 0 10, 10 10, 10 0, 0 0))'::geometry, 'GEOMETRYCOLLECTION (POINT(-1 0), LINESTRING(4 4,5 5))'::geometry);

-- CIP - collection within polygon
SELECT 'containsproperly210', ST_containsproperly('POLYGON((0 0, 0 10, 10 10, 10 0, 0 0))'::geometry, 'GEOMETRYCOLLECTION (POINT(5 5),LINESTRING(1 1,2 2))'::geometry);
-- CIP - collection on edge of polygon
SELECT 'containsproperly211', ST_containsproperly('POLYGON((0 0, 0 10, 10 10, 10 0, 0 0))'::geometry, 'GEOMETRYCOLLECTION (POINT(10 10), LINESTRING(0 0, 0 10))'::geometry);
-- CIP - collection outside polygon
SELECT 'containsproperly212', ST_containsproperly('POLYGON((0 0, 0 10, 10 10, 10 0, 0 0))'::geometry, 'GEOMETRYCOLLECTION (POINT(-1 0),LINESTRING(11 5,12 5))'::geometry);
-- CIP - elements of the collection fully outside and fully inside polygon
SELECT 'containsproperly213', ST_containsproperly('POLYGON((0 0, 0 10, 10 10, 10 0, 0 0))'::geometry, 'GEOMETRYCOLLECTION (POINT(-1 0), LINESTRING(4 4,5 5))'::geometry);

-- CIP - collection within polygon
SELECT 'overlaps210', ST_overlaps('POLYGON((0 0, 0 10, 10 10, 10 0, 0 0))'::geometry, 'GEOMETRYCOLLECTION (POINT(5 5),LINESTRING(1 1,2 2))'::geometry);
-- CIP - collection on edge of polygon
SELECT 'overlaps211', ST_overlaps('POLYGON((0 0, 0 10, 10 10, 10 0, 0 0))'::geometry, 'GEOMETRYCOLLECTION (POINT(10 10), LINESTRING(0 0, 0 10))'::geometry);
-- CIP - collection outside polygon
SELECT 'overlaps212', ST_overlaps('POLYGON((0 0, 0 10, 10 10, 10 0, 0 0))'::geometry, 'GEOMETRYCOLLECTION (POINT(-1 0),LINESTRING(11 5,12 5))'::geometry);
-- CIP - elements of the collection fully outside and fully inside polygon
SELECT 'overlaps213', ST_overlaps('POLYGON((0 0, 0 10, 10 10, 10 0, 0 0))'::geometry, 'GEOMETRYCOLLECTION (POINT(-1 0), LINESTRING(4 4,5 5))'::geometry);

-- CIP - collection within polygon
SELECT 'covers210', ST_covers('POLYGON((0 0, 0 10, 10 10, 10 0, 0 0))'::geometry, 'GEOMETRYCOLLECTION (POINT(5 5),LINESTRING(1 1,2 2))'::geometry);
-- CIP - collection on edge of polygon
SELECT 'covers211', ST_covers('POLYGON((0 0, 0 10, 10 10, 10 0, 0 0))'::geometry, 'GEOMETRYCOLLECTION (POINT(10 10), LINESTRING(0 0, 0 10))'::geometry);
-- CIP - collection outside polygon
SELECT 'covers212', ST_covers('POLYGON((0 0, 0 10, 10 10, 10 0, 0 0))'::geometry, 'GEOMETRYCOLLECTION (POINT(-1 0),LINESTRING(11 5,12 5))'::geometry);
-- CIP - elements of the collection fully outside and fully inside polygon
SELECT 'covers213', ST_covers('POLYGON((0 0, 0 10, 10 10, 10 0, 0 0))'::geometry, 'GEOMETRYCOLLECTION (POINT(-1 0), LINESTRING(4 4,5 5))'::geometry);

-- CIP - collection within polygon
SELECT 'coveredby210', ST_coveredby('POLYGON((0 0, 0 10, 10 10, 10 0, 0 0))'::geometry, 'GEOMETRYCOLLECTION (POINT(5 5),LINESTRING(1 1,2 2))'::geometry);
-- CIP - collection on edge of polygon
SELECT 'coveredby211', ST_coveredby('POLYGON((0 0, 0 10, 10 10, 10 0, 0 0))'::geometry, 'GEOMETRYCOLLECTION (POINT(10 10), LINESTRING(0 0, 0 10))'::geometry);
-- CIP - collection outside polygon
SELECT 'coveredby212', ST_coveredby('POLYGON((0 0, 0 10, 10 10, 10 0, 0 0))'::geometry, 'GEOMETRYCOLLECTION (POINT(-1 0),LINESTRING(11 5,12 5))'::geometry);
-- CIP - elements of the collection fully outside and fully inside polygon
SELECT 'coveredby213', ST_coveredby('POLYGON((0 0, 0 10, 10 10, 10 0, 0 0))'::geometry, 'GEOMETRYCOLLECTION (POINT(-1 0), LINESTRING(4 4,5 5))'::geometry);

-- CIP - collection within polygon
SELECT 'crosses210', ST_crosses('POLYGON((0 0, 0 10, 10 10, 10 0, 0 0))'::geometry, 'GEOMETRYCOLLECTION (POINT(5 5),LINESTRING(1 1,2 2))'::geometry);
-- CIP - collection on edge of polygon
SELECT 'crosses211', ST_crosses('POLYGON((0 0, 0 10, 10 10, 10 0, 0 0))'::geometry, 'GEOMETRYCOLLECTION (POINT(10 10), LINESTRING(0 0, 0 10))'::geometry);
-- CIP - collection outside polygon
SELECT 'crosses212', ST_crosses('POLYGON((0 0, 0 10, 10 10, 10 0, 0 0))'::geometry, 'GEOMETRYCOLLECTION (POINT(-1 0),LINESTRING(11 5,12 5))'::geometry);
-- CIP - elements of the collection fully outside and fully inside polygon
SELECT 'crosses213', ST_crosses('POLYGON((0 0, 0 10, 10 10, 10 0, 0 0))'::geometry, 'GEOMETRYCOLLECTION (POINT(-1 0), LINESTRING(4 4,5 5))'::geometry);

-- CIP - collection within polygon
SELECT 'touches210', ST_touches('POLYGON((0 0, 0 10, 10 10, 10 0, 0 0))'::geometry, 'GEOMETRYCOLLECTION (POINT(5 5),LINESTRING(1 1,2 2))'::geometry);
-- CIP - collection on edge of polygon
SELECT 'touches211', ST_touches('POLYGON((0 0, 0 10, 10 10, 10 0, 0 0))'::geometry, 'GEOMETRYCOLLECTION (POINT(10 10), LINESTRING(0 0, 0 10))'::geometry);
-- CIP - collection outside polygon
SELECT 'touches212', ST_touches('POLYGON((0 0, 0 10, 10 10, 10 0, 0 0))'::geometry, 'GEOMETRYCOLLECTION (POINT(-1 0),LINESTRING(11 5,12 5))'::geometry);
-- CIP - elements of the collection fully outside and fully inside polygon
SELECT 'touches213', ST_touches('POLYGON((0 0, 0 10, 10 10, 10 0, 0 0))'::geometry, 'GEOMETRYCOLLECTION (POINT(-1 0), LINESTRING(4 4,5 5))'::geometry);

-- CIP - collection within polygon
SELECT 'disjoint210', ST_disjoint('POLYGON((0 0, 0 10, 10 10, 10 0, 0 0))'::geometry, 'GEOMETRYCOLLECTION (POINT(5 5),LINESTRING(1 1,2 2))'::geometry);
-- CIP - collection on edge of polygon
SELECT 'disjoint211', ST_disjoint('POLYGON((0 0, 0 10, 10 10, 10 0, 0 0))'::geometry, 'GEOMETRYCOLLECTION (POINT(10 10), LINESTRING(0 0, 0 10))'::geometry);
-- CIP - collection outside polygon
SELECT 'disjoint212', ST_disjoint('POLYGON((0 0, 0 10, 10 10, 10 0, 0 0))'::geometry, 'GEOMETRYCOLLECTION (POINT(-1 0),LINESTRING(11 5,12 5))'::geometry);
-- CIP - elements of the collection fully outside and fully inside polygon
SELECT 'disjoint213', ST_disjoint('POLYGON((0 0, 0 10, 10 10, 10 0, 0 0))'::geometry, 'GEOMETRYCOLLECTION (POINT(-1 0), LINESTRING(4 4,5 5))'::geometry);

-- CIP - collection within polygon
SELECT 'relate210', ST_relate('POLYGON((0 0, 0 10, 10 10, 10 0, 0 0))'::geometry, 'GEOMETRYCOLLECTION (POINT(5 5),LINESTRING(1 1,2 2))'::geometry);
-- CIP - collection on edge of polygon
SELECT 'relate211', ST_relate('POLYGON((0 0, 0 10, 10 10, 10 0, 0 0))'::geometry, 'GEOMETRYCOLLECTION (POINT(10 10), LINESTRING(0 0, 0 10))'::geometry);
-- CIP - collection outside polygon
SELECT 'relate212', ST_relate('POLYGON((0 0, 0 10, 10 10, 10 0, 0 0))'::geometry, 'GEOMETRYCOLLECTION (POINT(-1 0),LINESTRING(11 5,12 5))'::geometry);
-- CIP - elements of the collection fully outside and fully inside polygon
SELECT 'relate213', ST_relate('POLYGON((0 0, 0 10, 10 10, 10 0, 0 0))'::geometry, 'GEOMETRYCOLLECTION (POINT(-1 0), LINESTRING(4 4,5 5))'::geometry);

-- CIP - collection within polygon
SELECT 'relate_pattern210', ST_relate('POLYGON((0 0, 0 10, 10 10, 10 0, 0 0))'::geometry, 'GEOMETRYCOLLECTION (POINT(5 5),LINESTRING(1 1,2 2))'::geometry, 'FF*FF****');
-- CIP - collection on edge of polygon
SELECT 'relate_pattern211', ST_relate('POLYGON((0 0, 0 10, 10 10, 10 0, 0 0))'::geometry, 'GEOMETRYCOLLECTION (POINT(10 10), LINESTRING(0 0, 0 10))'::geometry, 'FF*FF****');
-- CIP - collection outside polygon
SELECT 'relate_pattern212', ST_relate('POLYGON((0 0, 0 10, 10 10, 10 0, 0 0))'::geometry, 'GEOMETRYCOLLECTION (POINT(-1 0),LINESTRING(11 5,12 5))'::geometry, 'FF*FF****');
-- CIP - elements of the collection fully outside and fully inside polygon
SELECT 'relate_pattern213', ST_relate('POLYGON((0 0, 0 10, 10 10, 10 0, 0 0))'::geometry, 'GEOMETRYCOLLECTION (POINT(-1 0), LINESTRING(4 4,5 5))'::geometry, 'FF*FF****');

-- equals - element inside polygon
SELECT 'equals210', ST_equals('POLYGON((0 0, 0 10, 10 10, 10 0, 0 0))'::geometry,
'GEOMETRYCOLLECTION (POLYGON((0 0, 10 0, 10 10, 0 10, 0 0)),LINESTRING(0 2, 0 5))'::geometry);
-- equals - element on border of polygon
SELECT 'equals211', ST_equals('POLYGON((0 0, 0 10, 10 10, 10 0, 0 0))'::geometry,
'GEOMETRYCOLLECTION (POLYGON((0 0, 10 0, 10 10, 0 10, 0 0)),MULTIPOINT(0 2, 0 5))'::geometry);
-- equals - element on border and inside polygon
SELECT 'equals212', ST_equals('POLYGON((0 0, 0 10, 10 10, 10 0, 0 0))'::geometry,
'GEOMETRYCOLLECTION (POLYGON((0 0, 10 0, 10 10, 0 10, 0 0)),LINESTRING(0 2, 0 5, 5 5))'::geometry);
SELECT 'equals213', ST_equals('POLYGON((0 0, 0 10, 10 10, 10 0, 0 0))'::geometry,
'GEOMETRYCOLLECTION (POLYGON((0 0, 10 0, 10 10, 0 10, 0 0)),MULTIPOINT(0 2, 5 5))'::geometry);

-- #4299
SELECT '#4299', ST_Disjoint(ST_GeneratePoints(g, 1000), ST_GeneratePoints(g, 1000))
FROM (SELECT 'POLYGON((0 0,1 0,1 1,0 1,0 0))'::geometry AS g) AS f;

-- #4304
SELECT '#4304', ST_Equals(ST_GeneratePoints(g, 1000, 12345), ST_GeneratePoints(g, 1000, 12345)),
ST_Disjoint(ST_GeneratePoints(g, 1000, 12345), ST_GeneratePoints(g, 1000, 54321)),
ST_Disjoint(ST_GeneratePoints(g, 1000, 12345), ST_GeneratePoints(g, 1000)),
ST_Distance(ST_GeometryN(ST_GeneratePoints(g, 1000, 12345), 1000), 'POINT(0.801167838758 0.345281131175)'::geometry) < 1e-11
FROM (SELECT 'POLYGON((0 0,1 0,1 1,0 1,0 0))'::geometry AS g) AS f;

-- #4331
SELECT '#4304a', ST_3DMakeBox(ST_GeomFromText('POINT EMPTY',4326), ST_GeomFromText('POINT EMPTY',4326)) IS NULL;
SELECT '#4304b', ST_MakeBox2D(ST_GeomFromText('POINT EMPTY',4326), ST_GeomFromText('POINT EMPTY',4326)) IS NULL;

SELECT '#4176', ST_Intersects('POLYGON((0 0, 10 10, 3 5, 0 0))', 'GEOMETRYCOLLECTION(POINT(10 10), LINESTRING(0 0, 3 3))');

with mj as (select 'POINT(0 0)'::geometry geom) select '#4394' from mj a full join mj b on a.geom = b.geom;

-- #4445 Geometry comparisons
WITH p0 AS (SELECT 'POINT(0 0)'::geometry geom),
     p1 AS (SELECT 'POINT(1 1)'::geometry geom)
SELECT '#4445',
       p0.geom  <    p0.geom,
       p0.geom  <=   p0.geom,
       p0.geom  =    p0.geom,
       p0.geom  >=   p0.geom,
       p0.geom  >    p0.geom,
       p0.geom  <    p1.geom,
       p0.geom  <=   p1.geom,
       p0.geom  =    p1.geom,
       p0.geom  >=   p1.geom,
       p0.geom  >    p1.geom,
       p1.geom  <    p0.geom,
       p1.geom  <=   p0.geom,
       p1.geom  =    p0.geom,
       p1.geom  >=   p0.geom,
       p1.geom  >    p0.geom
FROM p0, p1;



WITH geom AS (
	SELECT 'TRIANGLE((0 0, 1 1, 0 1, 0 0))'::geometry geom
	union all
	SELECT 'TIN(((0 0, 1 1, 0 1, 0 0)))'::geometry geom
	union all
	SELECT 'TRIANGLE EMPTY'::geometry geom
)
select '#4399', 'ST_AsBinary', ST_AsBinary(geom)::text from geom
union all
select '#4399', 'ST_AsEWKB', ST_AsEWKB(geom)::text from geom
union all
select '#4399', 'ST_AsEWKT', ST_AsEWKT(geom)::text from geom
union all
select '#4399', 'ST_AsGML', ST_AsGML(3,geom)::text from geom
union all
select '#4399', 'ST_AsHEXEWKB', ST_AsHEXEWKB(geom)::text from geom
union all
select '#4399', 'ST_AsKML', ST_AsKML(ST_SetSRID(geom, 4326))::text from geom
union all
select '#4399', 'ST_AsText', ST_AsText(geom)::text from geom
union all
select '#4399', 'ST_AsTWKB', ST_AsTWKB(geom)::text from geom
union all
select '#4399', 'ST_AsX3D', ST_AsX3D(geom)::text from geom
union all
select '#4399', 'ST_GeoHash', ST_GeoHash(geom)::text from geom
union all
select '#4399', 'ST_AsGeoJSON', ST_AsGeoJSON(geom)::text from geom;

SELECT '#4599-1', ST_AsEWKT(ST_AddPoint(ST_GeomFromEWKT('LINESTRING(0 0 1, 1 1 1)'), ST_MakePoint(1, 2, 3)));
SELECT '#4599-2', ST_AsEWKT(ST_AddPoint(ST_GeomFromEWKT('LINESTRING(0 0 1, 1 1 1)'), ST_MakePoint(1, 2, 3), 0));
SELECT '#4599-3', ST_AsEWKT(ST_AddPoint(ST_GeomFromEWKT('LINESTRING(0 0 1, 1 1 1)'), ST_MakePoint(1, 2, 3), -1));

SELECT '#4670-0', ST_AsEWKT(ST_AddPoint('LINESTRING(0 0, 1 1, 3 3, 4 4)'::geometry, 'POINT(2 2)'::geometry, 0));
SELECT '#4670-1', ST_AsEWKT(ST_AddPoint('LINESTRING(0 0, 1 1, 3 3, 4 4)'::geometry, 'POINT(2 2)'::geometry, 1));
SELECT '#4670-2', ST_AsEWKT(ST_AddPoint('LINESTRING(0 0, 1 1, 3 3, 4 4)'::geometry, 'POINT(2 2)'::geometry, 2));
SELECT '#4670-3', ST_AsEWKT(ST_AddPoint('LINESTRING(0 0, 1 1, 3 3, 4 4)'::geometry, 'POINT(2 2)'::geometry, 3));
SELECT '#4670-4', ST_AsEWKT(ST_AddPoint('LINESTRING(0 0, 1 1, 3 3, 4 4)'::geometry, 'POINT(2 2)'::geometry, 4));
SELECT '#4670-5', ST_AsEWKT(ST_AddPoint('LINESTRING(0 0, 1 1, 3 3, 4 4)'::geometry, 'POINT(2 2)'::geometry, 5));

SELECT '#4689', _ST_DistanceTree('POLYGON ((30 10, 40 40, 20 40, 30 10))'::geography, 'POLYGON((81 6,140 35,-70 18,-51 0,-60 -46,106 -6,81 6))');

-- Polar Stereographic Axis order issues
SELECT '#4748', ST_AsEWKT(ST_Transform(ST_SetSRID(ST_Point(-36.75, -54.25), 4326), 3031),1);
SELECT '#4842', ST_AsEWKT(ST_Transform(ST_SetSRID(ST_Point(12.572, 66.081), 4326), 3413),1);

SELECT '#4718',
	round(degrees(
	ST_Azimuth('POINT(77.46412 37.96999)'::geography,
           'POINT(77.46409 37.96999)'::geography
           ))::numeric,3),
	round(degrees(
	ST_Azimuth('POINT(77.46412 37.96999)'::geography,
           'POINT(77.46429 37.96999)'::geography
           ))::numeric,3);

SELECT '#4727', _ST_DistanceTree('SRID=4326;POLYGON((-179.9 -85.05112877980659, -179.9 74.99999999999997, -152 80, -130 84.99999999999997, -115 85.05112877980659, -60 85.05112877980659, -60 79, -70 70, -130 50, -80 6, -65 -53, -100 -85.05112877980659, -179.9 -85.05112877980659))'::geography,
				ST_MakePoint(-150,-40), 0.0, true);

SELECT '#4796', st_astext(st_snaptogrid(st_normalize(st_simplifypreservetopology('MULTISURFACE(((178632.044 397744.007,178631.118 397743.786,178646.399 397679.574,178693.864 397690.889,178698.958 397669.487,178700.206 397669.784,178758.532 397683.689,178748.351 397726.468,178752.199 397727.384,178748.782 397741.904,178744.897 397740.98,178738.157 397769.303,178632.044 397744.007)))'::geometry,1)),1));

SELECT '#4812', st_srid('SRID=999999;POINT(1 1)'::geometry);

SELECT
'#4840',
round(degrees(ST_azimuth(C,N)))  AS az_n,
round(degrees(ST_azimuth(C,NE))) AS az_ne,
round(degrees(ST_azimuth(C,E)))  AS az_e,
round(degrees(ST_azimuth(C,SE))) AS az_se,
round(degrees(ST_azimuth(C,S)))  AS az_s,
round(degrees(ST_azimuth(C,SW))) AS az_sw,
round(degrees(ST_azimuth(C,W)))  AS az_w,
round(degrees(ST_azimuth(C,NW))) AS az_nw
FROM (SELECT
'POINT(5 55)'::geography AS C,
'POINT(5 56)'::geography AS N,
'POINT(6 56)'::geography AS NE,
'POINT(6 55)'::geography AS E,
'POINT(6 54)'::geography AS SE,
'POINT(5 54)'::geography AS S,
'POINT(4 54)'::geography AS SW,
'POINT(4 55)'::geography AS W,
'POINT(4 56)'::geography AS NW ) points;

SELECT '#4853', ST_ClusterDBSCAN(geom,  eps := 0.000906495804256269, minpoints := 4) OVER() AS cid FROM (VALUES ('0101000020E6100000E4141DC9E5934B40D235936FB6193940'::geometry), ('0101000020E6100000C746205ED7934B40191C25AFCE193940'::geometry), ('0101000020E6100000C780ECF5EE934B40B6BE4868CB193940'::geometry), ('0101000020E6100000ABB2EF8AE0934B404451A04FE4193940'::geometry)) AS t(geom);

SELECT '#4844', ST_AsEWKT(ST_SnapToGrid(ST_Transform('SRID=3575;POINT(370182.35945313 -2213980.8213281)'::geometry,4326),0.001));

SELECT '#4863', st_contains(geometry, st_scale(st_orientedenvelope(geometry),
 'SRID=3857; POINT(0.8 0.8)', st_centroid(geometry))) from (select
 'SRID=3857; POLYGON((-141972.789895508 6755731.24770785,-141935.49511986
 6755733.56891884,-141934.403428904 6755716.1146343,-141971.698204552
 6755713.77835553,-141972.789895508 6755731.24770785))'::geometry as
 geometry) x;

-- New Zealand forward -- SRID=2193;POINT(1766289 5927325)
SELECT '#4949', 'NZ forward', ST_AsEWKT(ST_SnapToGrid(ST_Transform(
  'SRID=4326;POINT(174.863597538742 -36.785298415230315)'::geometry, 2193),0.1));
--- New Zealand inverse (opposite EPSG order) -- SRID=4326;POINT(174.863598 -36.785298)
SELECT '#4949', 'NZ inverse', ST_AsEWKT(ST_SnapToGrid(ST_Transform(
  'SRID=2193;POINT(1766289 5927325)'::geometry, 4326),0.000001));
-- British Columbia forward (respect EPSG order) -- SRID=3005;POINT(1286630.44 561883.98)
SELECT '#4949', 'BC forward', ST_AsEWKT(ST_SnapToGrid(ST_Transform(
  'SRID=4269;POINT(-122 50)'::geometry, 3005),0.01));
-- British Columbia inverse (respect EPSG order) -- SRID=4269;POINT(-122 50)
SELECT '#4949', 'BC inverse', ST_AsEWKT(ST_SnapToGrid(ST_Transform(
  'SRID=3005;POINT(1286630.44 561883.98)'::geometry, 4269),0.000001));
--  North Pole LAEA Europe inverse -- SRID=4326;POINT(19.4921659 69.7902258)
SELECT '#4949', 'North Pole LAEA inverse', ST_AsEWKT(ST_SnapToGrid(ST_Transform(
  'SRID=3575;POINT(370182 -2213980)'::geometry,4326),0.0000001));
-- Polar Stereographic forward -- SRID=3413;POINT(2218082.1 -1409150)
SELECT '#4949', 'Arctic Stereographic forward', ST_AsEWKT(ST_SnapToGrid(ST_Transform(
  'SRID=4326;POINT(12.572160  66.081084)'::geometry,3413),0.1));
-- Antarctic Polar Stereographic -- SRID=3031;POINT(-2399498.7 3213318.5)
SELECT '#4949', 'Antarctic Stereographic forward', ST_AsEWKT(ST_SnapToGrid(ST_Transform(
  'SRID=4326;POINT(-36.75 -54.25)'::geometry, 3031),0.1));

-- #4916, #4770, #4724, #4916, #4940
SELECT '#4770.a',
 ST_Union(NULL::geometry) OVER (ORDER BY b)
FROM (VALUES ('A0006', 300),
	         ('A0006', 302)) t(a, b);

WITH w AS (
  SELECT
    ST_Union(g) OVER (PARTITION BY b ORDER BY a) AS g,
    Sum(b) OVER (PARTITION BY b ORDER BY a) AS s
  FROM (VALUES ('POINT(0 0)'::geometry, 'A0006', 300),
  	           ('POINT(1 1)'::geometry, 'A0006', 302)) t(g, a, b)
)
SELECT '#4770.b', ST_AsText(g), s FROM w;

WITH w AS (
  SELECT
    ST_Union(g) OVER (PARTITION BY a ORDER BY b) AS g,
    Sum(b) OVER (PARTITION BY a ORDER BY b) AS s
  FROM (VALUES ('POINT(0 0)'::geometry, 'A0006', 300),
  	           ('POINT(1 1)'::geometry, 'A0006', 302)) t(g, a, b)
)
SELECT '#4770.c', ST_AsText(g), s FROM w;

-- https://trac.osgeo.org/postgis/ticket/4799
SELECT
    '#4799', ST_AsGeoJSON(data.*, geom_column => 'geom2', maxdecimaldigits => 3)
FROM
    (SELECT
        1 AS id,
        ST_SnapToGrid(ST_Transform(geom, 3035), 1) geom1,
        ST_SnapToGrid(ST_Transform(geom, 25832), 1) geom2
    FROM
        ST_SetSRID(ST_MakePoint(7, 51), 4326) geom
    ) data;


-- https://trac.osgeo.org/postgis/ticket/5008
SELECT
	'#5008',
	ST_DWithin('POINT EMPTY', 'POINT(0 0)', 'Inf'),
	ST_DWithin('POINT(1 1)', 'POLYGON EMPTY', 'Inf');

-- https://trac.osgeo.org/postgis/ticket/5151
SELECT '#5151', ST_SetPoint(ST_GeomFromText('LINESTRING EMPTY',4326), 1, ST_GeomFromText('POINT(40 50)',4326)) As result;
