#!/usr/bin/env bash

set -o errexit
set -o nounset
set -o pipefail

declare -r NODENAME="$1"
declare -r PGVER="$2"
declare -r PGDATA="$3"
declare -r PRIM_IP="$4"
declare -r PRIM_NODE="$5"

declare -r CUSTOMDIR="${PGDATA}/conf.d"

# cleanup
systemctl --quiet --now disable "postgresql-${PGVER}"
rm -rf "${PGDATA}"

if [ "$NODENAME" == "$PRIM_NODE" ]; then
    # init instance
    PGSETUP_INITDB_OPTIONS="--data-checksums"
    "/usr/pgsql-${PGVER}/bin/postgresql-${PGVER}-setup" initdb

    # pg_hba setup
    cat<<-EOC > "${PGDATA}/pg_hba.conf"
	local all         all                      trust
	host  all         all      0.0.0.0/0       trust

	# forbid self-replication
	local replication all                      reject
	host  replication all ${NODENAME}          reject
	host  replication all 127.0.0.1/32         reject
	host  replication all ::1/128              reject
	host  replication all ${PRIM_IP}/32        reject

	# allow any standby connection
	host  replication postgres 0.0.0.0/0       trust
	EOC

    # postgresql.conf setup
    mkdir -p "$CUSTOMDIR"
    echo "include_dir = 'conf.d'" >> "${PGDATA}/postgresql.conf"

    cat <<-EOC > "${CUSTOMDIR}/cluster_name.conf"
	cluster_name = 'pgsql-$NODENAME'
	EOC

    cat <<-'EOC' > "${CUSTOMDIR}/custom.conf"
	listen_addresses = '*'
	port = 5432
	wal_level = replica
	max_wal_senders = 10
	hot_standby = on
	hot_standby_feedback = on
	log_destination = 'syslog,stderr'
	log_checkpoints = on
	log_min_duration_statement = 0
	log_autovacuum_min_duration = 0
	log_replication_commands = on
	EOC

    if [ "${PGVER%%.*}" -lt 13 ]; then
        # recovery.conf setup
        cat <<-'EOC' >> "${CUSTOMDIR}/custom.conf"
		wal_keep_segments = 64
		EOC
    else
        cat <<-'EOC' >> "${CUSTOMDIR}/custom.conf"
		wal_keep_size = 1GB
		EOC
    fi

    if [ "${PGVER%%.*}" -lt 12 ]; then
        # recovery.conf setup
        cat<<-EOC > "${CUSTOMDIR}/recovery.conf.pcmk"
		standby_mode = on
		primary_conninfo = 'host=${PRIM_IP} application_name=${NODENAME}'
		recovery_target_timeline = 'latest'
		EOC
    else
        cat <<-EOC > "${CUSTOMDIR}/repli.conf"
		primary_conninfo = 'host=${PRIM_IP} application_name=${NODENAME}'
		EOC

        # standby_mode disappear in v12
        # no need to add recovery_target_timeline as its default is 'latest'
        # since v12
    fi

    # backing up files
    cp "${PGDATA}/pg_hba.conf"        "${PGDATA}/.."
    cp "${PGDATA}/postgresql.conf"    "${PGDATA}/.."
    cp "${CUSTOMDIR}"/*               "${PGDATA}/.."

    chown -R postgres:postgres "$PGDATA"

    # create master ip
    ip -o addr show to "${PRIM_IP}" | if ! grep -q "${PRIM_IP}"
    then
        DEV=$(ip route show to "${PRIM_IP}/24"|grep -Eo 'dev \w+')
        ip addr add "${PRIM_IP}/24" dev "${DEV/dev }"
    fi

    # restart master pgsql
    systemctl --quiet start "postgresql-${PGVER}"

    exit
fi

# building standby

# wait for the primary to listen
while ! "/usr/pgsql-${PGVER}/bin/pg_isready" -qh "${PRIM_IP}"; do sleep 1 ; done


# build standby
"/usr/pgsql-${PGVER}/bin/pg_basebackup" -h "${PRIM_IP}" -U postgres \
    -D "${PGDATA}" -X stream

# set pg_hba
cat<<EOC > "${PGDATA}/pg_hba.conf"
local all         all                      trust
host  all         all      0.0.0.0/0       trust

# forbid self-replication
local replication all                      reject
host  replication all ${NODENAME}          reject
host  replication all 127.0.0.1/32         reject
host  replication all ::1/128              reject
host  replication all ${PRIM_IP}/32        reject

# allow any standby connection
host  replication postgres 0.0.0.0/0       trust
EOC

cat <<EOC > "${CUSTOMDIR}/cluster_name.conf"
cluster_name = 'pgsql-$NODENAME'
EOC

if [ "${PGVER%%.*}" -lt 12 ]; then
    # recovery.conf setup
    cat<<-EOC > "${CUSTOMDIR}/recovery.conf.pcmk"
	standby_mode = on
	primary_conninfo = 'host=${PRIM_IP} application_name=${NODENAME}'
	recovery_target_timeline = 'latest'
	EOC

    cp "${CUSTOMDIR}/recovery.conf.pcmk" "${PGDATA}/recovery.conf"
else
    cat <<-EOC > "${CUSTOMDIR}/repli.conf"
	primary_conninfo = 'host=${PRIM_IP} application_name=${NODENAME}'
	EOC

    # standby_mode disappear in v12
    # no need to add recovery_target_timeline as its default is 'latest' since v12
    touch "${PGDATA}/standby.signal"
fi

# backing up files
cp "${PGDATA}/pg_hba.conf"        "${PGDATA}/.."
cp "${PGDATA}/postgresql.conf"    "${PGDATA}/.."
cp "${CUSTOMDIR}"/*               "${PGDATA}/.."

chown -R "postgres:postgres" "${PGDATA}/.."

# start
systemctl --quiet start "postgresql-${PGVER}"
